/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: defaultwindowingpolicy.h,v 1.5 2004/06/15 12:56:45 jd Exp $

   Created 05/14/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file leg/libs/window/defaultwindowingpolicy.h
  \brief default windowing policy.
*/

#ifndef LEG_LIBS_WINDOW_DEFAULT_WINDOWING_POLICY_H
#define LEG_LIBS_WINDOW_DEFAULT_WINDOWING_POLICY_H

#include "leg/support/states/states.h"
#include "inputs.h"
#include "states.h"
#include "actorpolicy.h"
#include "reactorpolicy.h"
#include "leg/support/graphics/viewmode.h"
#include "leg/support/graphics/glconfig.h"
#include "leg/support/timing/timing.h"
#include "leg/libs/threads/sync.h"

namespace leg
{
namespace libs
{
namespace window
{

using leg::support::graphics::ViewMode;
using leg::support::graphics::projection;
using leg::support::graphics::modelview;
using leg::support::graphics::texture;

//! Default policy for windowing.
/*!
 * A generic policy for handling window that is assured to work in a default
 * configuration (with all default templates providen with leg).
 * A windowing policy is intended to manage the functions of the window, and
 * permits to draw into it.
 *
 * template arguments description:
 * 	* Win represents the concrete type of the window (this will be
 * 	generally leg::support::window::AbstractWindow<OSWindow>).
 * 	* Connection is the display connector type.
 * 	* Sync is a thread synchronizer.
 * 	* StateMachine is a state machine (to be a member) for the Window.
 *
 * status:
 * 	in progress.
 *
 * info:
 * 	none.
 *
 * @sa GenericWindow.
 */
template
<
   class Win,
   class Connection,
   class Sync = leg::libs::threads::Sync<>,
   class StateMachine = leg::support::states::StateMachine< Win,Input,State,Sync,ActorPolicy,ReactorPolicy> 
>
class DefaultWindowingPolicy:	 public Win,
				 public Sync
{
   protected:
   
   typedef Win				  ParentType;
   typedef typename ParentType::Attr	  Attributes;
   typedef typename ParentType::Drawable  Drawable;

   using ParentType::attr;
   using ParentType::display;
   
   StateMachine			 *state_machine;

   DefaultWindowingPolicy();
   
   public:
   
   //! As almost all the windowing stuff, the constructor needs a reference to
   //  the display connection, that needs to be already opened.
   explicit DefaultWindowingPolicy (Connection& dc):  ParentType (dc),
						      Sync(),
						      state_machine (0)
   {
      attr.SetKeyboardAutoRepeatMode (display.Get(), false);
      state_machine = new StateMachine (state::destructed);
      Act (input::construct);
   }

   ~DefaultWindowingPolicy()
   {
      Act (input::destruct);
      attr.SetKeyboardAutoRepeatMode (display.Get(), true);
      delete state_machine;
      state_machine = 0;
   }

   void
   Create()
   {
      ParentType::Create();
      Act (input::create);
      if (ParentType::GetAttributes().GetFullScreenUse())
	 Act (input::fullscreen);
      else
	 Act (input::window);
   }
   
   void Destroy()
   {
      ParentType::Destroy();
      Act (input::destroy);
   }
   
   using ParentType::SwitchToFullScreen;
   using ParentType::SwitchToWindow;
   using ParentType::GetAttributes;
   using ParentType::GetDrawable;

   using Sync::Lock;
   using Sync::Unlock;

   void
   Act (const Input& input)
   {
      state_machine->Act (input);
   }

   void
   React()
   {
      state_machine->React (*this);
   }
};

}
}
}
#endif
