/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: timer.h,v 1.1 2004/05/13 04:16:02 jechk Exp $
   $Log: timer.h,v $
   Revision 1.1  2004/05/13 04:16:02  jechk
   Split timing into two parts; moved message to libs.

   Revision 1.5  2004/04/30 20:15:54  jechk
   Big merge.  See ChangeLog for details.

   Revision 1.4  2004/03/11 05:53:40  jechk
   Fixed a problem in test_timing.c.

   Revision 1.3  2004/03/09 03:54:57  jechk
   Made support::timing::Timer::timing private, minor changes.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 2/21/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/timing/timer.h
  \brief The Timer class, a thread that is created at certain times.
*/

#include <leg/libs/timing/timing.h>
#include <leg/support/threads/threads.h>

#ifndef LEG_LIBS_TIMING_TIMER_H
#define LEG_LIBS_TIMING_TIMER_H

namespace leg
{

namespace libs
{

namespace timing
{

class Timer : public support::threads::Thread
{
public:
  //! Creates a timer for the specified timing.
  /*! The timer will not go off unless set.
   */
  Timer (Timing &timing);
  virtual ~Timer ();

  //! Sets the timer to go off at the specified game time.
  /*!
   * Returns an int identifying the timer.  It is possible for a timer
   * to be multiply set.  @arg data is passed to the thread's Go()
   * function. 
   */
  int Set (support::timing::game_time when, void *data = NULL);

  //! Sets the timer to go off repeatedly.
  /*!
   * The timer will go off at @arg when, and will then repeat every
   * @arg interval units.  It is possible for a timer to be multiply
   * set.  Returns an int identifying the timer.  @arg data is passed
   * to the thread's Go() function. 
   */
  int SetRepeating (support::timing::game_time when, support::timing::game_time interval,
		    void *data = NULL);

  //! Removes the setting corresponding to the identifier.
  void Unset (int identifier);

  //! Accesses the timing object controlling the timer.
  Timing& GetTiming () const
  {
    return timing;
  }

private:
  // The timing object controlling the timer.
  Timing &timing;

  // Called periodically with a new time.
  void Ping (struct timeval tv);

  struct TimerSetting
  {
    TimerSetting (int identifier, struct timeval when, void *data);
    TimerSetting (int identifier, struct timeval when,
		  struct timeval interval, void *data);

    int identifier;
    struct timeval when;
    bool is_repeating;
    struct timeval interval;

    bool active;

    void *data;
  };

  std::list<TimerSetting> settings;
  int next_identifier;

  friend class Timing::TimerThread;
};

}

}

}

#endif // LEG_SUPPORT_TIMING_TIMER_H
