/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: defaultsettingpolicy.h,v 1.5 2004/06/06 02:09:01 jd Exp $

   Created 05/21/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/threads/defaultsettingpolicy.h
  \brief default setting policy for threader cells.
*/

#ifndef LEG_LIBS_THREADS_DEFAULT_SETTING_POLICY_H
#define LEG_LIBS_THREADS_DEFAULT_SETTING_POLICY_H

#include "leg/support/threads/common.h"
#include "leg/libs/threads/sync.h"
#include "leg/support/states/states.h"
#include "inputs.h"
#include "states.h"
#include "actorpolicy.h"
#include "reactorpolicy.h"

namespace leg
{
namespace libs
{
namespace threads
{

using leg::support::threads::ThreadCancelState;
using leg::support::threads::enable;
using leg::support::threads::disable;
using leg::support::threads::ScheduleAlgorithm;
using leg::support::threads::normal;
using leg::support::threads::fifo;
using leg::support::threads::robin;

//! Default policy for threads setting.
/*!
 * A thread set is a set of behaviors for this thread. Any other policies to 
 * be replaced by this one, must respect this interface.
 * This policy defines several functions that set the behavior of the thread
 * in some situations: loading, pre-running, post-running, and surely others
 * in a proach future.
 * Generally, this default setting behavior will be suffisant for any
 * ThreaderCell in the whole lib.
 *
 * template argument description:
 *    * SyncCell is the cell class (the user class to be threaded) that must
 *    have synchronization capabilities (Mutex inheritence is recommanded).
 *    * NoSyncThread is a thread class without synchronization.
 * 
 * status:
 *    in progress.
 *
 * info:
 *    Any ThreaderCell that will use this policy will wait for a signal and
 *    its state changed into 'ready' when it loads.
 *
 * @sa ThreaderCell, Sync, Cond, Thread.
 */
template
<
   class SyncCell,
   class NoSyncThread,
   class StatyThread = support::states::HostInheritedStateMachine<NoSyncThread,
								  Input,
								  State,
								  libs::threads::Sync<>,
								  ActorPolicy,
								  ReactorPolicy> 
>
class DefaultSettingPolicy:public SyncCell,
			   public StatyThread
{
   protected:

   typedef typename SyncCell::MutexType MutexType;

   using StatyThread::Act;
   using StatyThread::React;
   using StatyThread::state_mutex;
   using StatyThread::SetCancelState;
 
   using SyncCell::Lock;
   using SyncCell::Unlock;
   using SyncCell::GetMutex;
   
   using StatyThread::Wait;
   using StatyThread::Signal;

   DefaultSettingPolicy ():SyncCell(),
			   StatyThread (state::undefined)
   {
      Act (input::constructed);
   }

   //! needed by ThreaderCell.
   explicit DefaultSettingPolicy (SyncCell& sync_cell):	 SyncCell (sync_cell),
							 StatyThread (state::undefined)
   {
      Act (input::constructed);
   }

   DefaultSettingPolicy (const DefaultSettingPolicy&);

   ~DefaultSettingPolicy()
   {
      Act (input::destructed);
   }

   const DefaultSettingPolicy&
   operator= (const DefaultSettingPolicy&);
   
   //! Load and runs the thread.
   /*!
    * Load the configuration of the thread (so this is here we could set the 
    * scheduling algorithm), and creates it (so runs it).
    */
   inline void
   Load ()
   {
      Lock();
      Act (input::locked);
      NoSyncThread::SetScheduleConfig (normal);
      Act (input::configured);
      NoSyncThread::Create();
      Act (input::created+input::cancelable+input::unlocked+input::loaded);
      Unlock();
   }

   inline void
   Unload ()
   {
   }

   //! PreRuning routine.
   /*
    * Generally, good threads will wait by using conditions.
    */
   inline void
   PreRun ()
   {
      Lock();
      SetCancelState (disable);
      Act (input::locked+input::not_cancelable);
      Unlock();
      
      WaitReady();
   }

   //! Run the thread.
   inline void*
   Run (void *args)
   {
      void *ret = 0;

      while (1){
	 state_mutex.Lock();
	 if (this->state & state::ending){
	    state_mutex.Unlock();
	    break;
	 }
	 state_mutex.Unlock();
	 
	 ret = SyncCell::Go (args);
	 if (ret){
	    break;
	 }
      
	 React();
      }
      
      Act (input::ended);
      
      return ret;
   }

   //! PostRunning routine.
   inline void
   PostRun ()
   {
      React();
   }

   public:

   void
   WaitReady()
   {
      while (1){
	 Lock();
	 Act (input::in_wait);
	 Wait (GetMutex());
	 state_mutex.Lock();
	 if (this->state & state::ready){
	    state_mutex.Unlock();
	    Act (input::run+input::unlocked);
	    Unlock();
	    break;
	 }
	 state_mutex.Unlock();
	 Unlock();
      }
   }

   void
   SignalReady()
   {
      Lock();
      Act (input::ready);
      Signal();
      Unlock();
   }
};

}
}
}
#endif
