/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 03/28/04 by J-D Frattini <zionarea@free.fr>
   
   Copyright (c) 2005 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/scene/simpleobject.h
  \brief a simple object.
*/

#ifndef LEG_LIBS_SCENE_SIMPLE_OBJECT_H
#define LEG_LIBS_SCENE_SIMPLE_OBJECT_H

#include "object.h"

namespace leg
{
namespace libs
{
namespace scene
{

class SimpleObject: public Object
{
   public:

   //
   // ! WARNING !
   //
   // You MUST redefine those two declarations for each class you will inherit
   // in this hierarchy. You also MUST to assign them with the appropriate 
   // values. Otherwise, expect data corruption on loading and saving.
   // 
   
   static const std::string main_id;
   static const std::string sub_id;

   SimpleObject();

   SimpleObject (const SimpleObject& o);

   void
   Init();

   void
   Update();

   SimpleObject*
   Clone();

   const std::string
   GetMainId()
   {
      return main_id;
   }

   const std::string
   GetSubId()
   {
      return sub_id;
   }

   const Entity::Propriety
   GetPropriety() const;

   protected:

   void
   Copy (const SimpleObject& o);
};

class SimpleCar: public SimpleObject
{
   public:

   static const std::string main_id;
   static const std::string sub_id;

   SimpleCar(): SimpleObject()
   {
   }
   
   const std::string
   GetMainId()
   {
      return main_id;
   }

   const std::string
   GetSubId()
   {
      return sub_id;
   }

   const Entity::Propriety
   GetPropriety() const
   {
      return moveable_object;
   }

   void
   Init()
   {
   }

   void
   Update()
   {
      // make a move regarding time u_clock and u_dt
      pos.x += 1. * u_dt;
   }

   SimpleCar*
   Clone()
   {
      return new SimpleCar (*this);
   }

   protected:

   void
   Copy (const SimpleObject& o)
   {
   }
};

class DynamicObject: public SimpleObject
{
   protected:
   
   libs::physics::Entity *physics;

   public:

   DynamicObject(): SimpleObject()
   {
   }

   DynamicObject (const DynamicObject& o):
      SimpleObject (static_cast<const SimpleObject&> (o))
   {
   }

   const DynamicObject&
   operator = (const DynamicObject& o)
   {
      SimpleObject::operator = (static_cast<const SimpleObject&> (o));
      Copy (o);

      return *this;
   }

//   void
//   SetVelocity (Vector& v);

   // and so on

   protected:

   void
   Copy (const DynamicObject& o)
   {
   }
};

class InteractiveObject: public DynamicObject
{
   public:

   InteractiveObject (): DynamicObject()
   {
   }

   InteractiveObject (const InteractiveObject& o):
      DynamicObject (static_cast<const DynamicObject&> (o))
   {
      Copy (o);
   }

   const InteractiveObject&
   operator = (const InteractiveObject& o)
   {
      DynamicObject::operator = (static_cast<const DynamicObject&> (o));
      Copy (o);

      return *this;
   }

   protected:

   void
   Copy (const InteractiveObject& o)
   {
   }
};

class ArticulatedObject: public InteractiveObject
{
   public:

   static const std::string main_id;
   static const std::string sub_id;

   ArticulatedObject(): InteractiveObject()
   {
   }

   ArticulatedObject (const ArticulatedObject& o):
      InteractiveObject (static_cast<const InteractiveObject&> (o))
   {
      Copy (o);
   }

   const ArticulatedObject&
   operator = (const ArticulatedObject& o)
   {
      InteractiveObject::operator = (static_cast<const InteractiveObject&> (o));
      Copy (o);

      return *this;
   }
   
   const std::string
   GetMainId()
   {
      return main_id;
   }

   const std::string
   GetSubId()
   {
      return sub_id;
   }

   void
   Init()
   {
   }

   void
   Update()
   {
   }

   const Entity::Propriety
   GetPropriety() const
   {
      return (articulated_object | dynamic_object);
   }

   ArticulatedObject*
   Clone()
   {
      return new ArticulatedObject (*this);
   }

   protected:

   void
   Copy (const ArticulatedObject& o)
   {
   }
};

}
}
}

#endif // LEG_LIBS_SCENE_SIMPLE_OBJECT_H
