/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 12/08/04 by J-D Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/scene/factory.h
  \brief object factory and registration.
*/

#ifndef LEG_LIBS_SCENE_FACTORY_H
#define LEG_LIBS_SCENE_FACTORY_H

#include "simplecamera.h"
#include "simpleobject.h"
#include "staticobject.h"
#include "pointlight.h"
#include "infinitelight.h"

#include <loki/Singleton.h>
#include <map>
#include <string>

namespace leg
{
namespace libs
{
namespace scene
{

//! The scene object factory
/*!
 * This class is intended to register and creates various object types from a
 * same hierarchy. It is mainly based on loki's factory. There's very few
 * difference between them.
 *
 * Each class type must register via the Register function.
 *
 * TAD:
 *    
 *    * Abstract is the base type of the aimed hierarchy.
 *
 *    * Id is the type used for storing identifiers.
 *
 *    * Creator is the class/function interface used to create objects.
 *
 * status:
 *    Safe.
 *
 * info:
 *    needs many other registrations.
 */
template
<
   class Abstract,
   class Id = std::string,
   class Creator = Abstract* (*)()
>
class Factory
{
   typedef std::pair<Id,Id>	 Tid;
   typedef std::map<Tid,Creator> Map;
   typedef std::map<Tid,bool>	 Reg;

   Map map;
   Reg reg;
   
   public:

   ~Factory()
   {
      map.clear();
      reg.clear();
   }

   inline bool
   Register (const Id& main_id, const Id& sub_id, Creator creator)
   {
      bool is_reg = map.insert (typename Map::value_type (Tid (main_id, sub_id), creator)).second;
      reg[Tid (main_id,sub_id)] = is_reg;
      
      return is_reg;
   }

   inline bool
   Unregister (const Id& main_id, const Id& sub_id)
   {
      bool v = map.erase (Tid (main_id, sub_id)) == 1;
      reg[Tid (main_id,sub_id)] = !v;
      
      return v;
   }

   inline bool
   IsRegistered (const Id& main_id, const Id& sub_id)
   {
      return reg[Tid (main_id, sub_id)];
   }

   inline Abstract*
   Create (const Id& main_id, const Id& sub_id)
   {
      typename Map::const_iterator i = map.find (Tid (main_id,sub_id));
      
      if (i != map.end())
	 return (i->second)();

      return 0;
   }
};

typedef Loki::SingletonHolder <Factory <Entity> >  ObjectFactory;

}
}
}

#endif // LEG_LIBS_SCENE_FACTORY_H
