/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: postoffice.h,v 1.1 2004/06/23 00:57:51 jd Exp $

   Created 06/22/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/message/postoffice.h
  \brief Post-Office message treater.
*/

#ifndef LEG_LIBS_MESSAGE_POST_OFFICE_H
#define LEG_LIBS_MESSAGE_POST_OFFICE_H

#include "common.h"
#include "basedeliverer.h"
#include "deliverer.h"
#include "leg/support/timing/timing.h"
#include "leg/support/threads/mutex.h"
#include <list>

namespace leg
{
namespace libs
{
namespace message
{

using support::timing::Timing;
using support::timing::GameTime; // or game_time or gametime or simply time or Time...
using support::threads::Mutex;

//! Leg's inter-object messaging post-office.
/*!
 * All leg's communication between classes is done by posting messages threw
 * this class static interface. However, for the moment, only destination 
 * classes that match the desired interface (see Deliverer) could receive the
 * messages send threw this class. Real communication between players could not
 * be done threw this class, but could take profit from this class with simply
 * inheriting from it (there might also need to inherit from Deliverer too).
 *
 * PostOffice is designed to be thread safe under some conditions: the member
 * function Actualize() should be called from a different thread than the
 * sending functions. However, for conveniency, Actualize() is not protected for
 * the moment; it will later, when the engine layer will be ready.
 *
 * status:
 *    experimenting.
 *
 * info:
 *    multithread safe.
 *
 * @sa Message,BaseDeliverer,Deliverer,Mutex.
 */
class PostOffice
{
   protected:

   PostOffice (const PostOffice&);
   
   const PostOffice& operator = (const PostOffice&);

   //! BaseDeliverer is known as Treater inside this class.
   typedef BaseDeliverer			Treater;

   //! A mail is a pair of a pointer to Treater and a message.
   typedef std::pair<Treater*,const Message>	Mail;

   //! A timed mail is a pair of a Mail and a game time.
   typedef std::pair<Mail,GameTime>		TimedMail;
   
   //! A mail queue is not a real queue (for conveniency) but a list.
   typedef std::list<TimedMail>			MailQueue;

   static Mutex		mutex;		  //!< threads-shared-memory synchronizer.
   
   static Timing	timing;		  //!< timing.

   static MailQueue	mail_queue;	  //!< mail list (queue).

   PostOffice();

   ~PostOffice();

   public:

   //! Send a message to an object.
   /*!
    * This directly send the message given in first argument to the object given
    * as the second argument without passing threw the mail queue. So, 
    * Actualize() will have no effect.
    *
    * info:
    *	 polymorphism is required.
    */
   template <class T>
   static void
   Send (const Message& msg, T& to)
   {
      Treater *treat = new Deliverer<T> (to);
      treat->Deliver (msg);
      delete treat;
   }

   //! Send a message in a certain time delay.
   /*!
    * This save the message and the matching sending time in the queue, but 
    * does not send it in reality. Actualize() must be called elsewhere in order
    * the message to be delivered.
    *
    * info:
    *	 none.
    */
   template <class T>
   static void
   SendIn (GameTime sec, const Message& msg, T& to)
   {
      mutex.Lock();
      Timing t = const_cast <const Timing&> (timing);
      Treater *treat = new Deliverer<T> (to);
      mail_queue.push_back (TimedMail (Mail (treat,msg),t.GetCurrentTime()+sec));
      mutex.Unlock();
   }

   //! Send a message at a certain time date.
   /*!
    * This saves the message and the time date in the queue but does not deliver
    * it to the object. Actualize() must be called elsewhere in order the 
    * message to be delivered.
    *
    * info:
    *	 none.
    */
   template <class T>
   static void
   SendAt (GameTime timing, const Message& msg, T& to)
   {
      mutex.Lock();
      Treater *treat = new Deliverer<T> (to);
      mail_queue.push_back (TimedMail (Mail (treat,msg),timing));
      mutex.Unlock();
   }

   //protected:
  
   //! Actualizes the awaiting messages in the queue.
   /*!
    * Delivers the messages to the correct destination object if their matching
    * time have expired. Let the message in the queue at the same position if
    * the date isn't passed. This function should be called from another thread
    * than the sending requests. Works too on same thread if necessary.
    *
    * status:
    *	 working.
    *
    * info:
    *	 should be protected for engine's layer inheritence could use it safely.
    */
   static void
   Actualize();
};

}
}
}

#endif
