/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: drawer.h,v 1.2 2004/07/15 23:24:36 jd Exp $

   Created 06/14/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/graphics/drawer.h
  \brief a generic drawer.
*/

#ifndef LEG_LIBS_GRAPHICS_DRAWER_H
#define LEG_LIBS_GRAPHICS_DRAWER_H

#include "leg/support/graphics/perspective.h"
#include "leg/support/graphics/ortho.h"
#include "leg/support/graphics/viewport.h"
#include "leg/support/graphics/viewmode.h"
#include "leg/support/graphics/defaultviewingpolicy.h" // should surely be in libs.
#include "leg/support/timing/timing.h"
#include "viewer.h"
#include "leg/support/utils/errors.h"

namespace leg
{
namespace libs
{
namespace graphics
{

using namespace leg::support::utils::error;
using support::graphics::Perspective;
using support::graphics::Ortho;
using support::graphics::Viewport;
using support::graphics::MatrixMode;
using support::graphics::DefaultViewingPolicy;

//! Default policy for drawing into a window.
/*!
 * Default drawing policy is intended to provide a default manner in order to
 * draw into a window with OpenGL commands. Generally, using this policy will
 * be suffisant for almost all users.
 * This policy is intended to be used threw libs::graphics::Drawer.
 *
 * tad:
 *    Window is the static type of window (this will generally be the type
 *    libs::Window<> because we need such functionalities support does not
 *    provide at all: Act, Lock...). Look at libs::window::Window.
 *    ViewingPolicy is the policy used for setting up the viewing into the
 *    window threw OpenGL.
 *    GLConfig is the class used to configure OpenGL (default is well).
 *
 * status:
 *    working.
 *
 * info:
 *    This class is not intended to be instanciated.
 *    Member functions should remain protected.
 *
 * TODO:
 *    It will be better if we could give the variable names of the window's
 *    Act() argument instead of any hard string ("blabla"): names may change
 *    during the development time, and it will become very difficult to know
 *    where all hard strings stand.
 *
 * @sa Drawer,ViewingPolicy.
 */
template
<
   class Window,
   class ViewingPolicy,
   class GLConfig = support::graphics::GLConfig
>
class DefaultDrawingPolicy: protected ViewingPolicy
{
   protected:

   typedef ViewingPolicy Viewing;

   GLConfig gl_config;	   //!< stored gl configuration.
   
   //! Timing management.
   /*!
    * Timing is used to store and manage the framerate status of the rendering
    * threw this policy.
    *
    * status:
    *	 working.
    *
    * info:
    *	 none.
    */
   struct Timing
   {
      float			    n,t;		 //!< frequency,period.
      unsigned int		    frame_counter;	 //!< counter of frames.
      unsigned int		    frames_last_second;	 //!< number of frames last second.
      leg::support::timing::Timing  clock;		 //!< clock.

      Timing():n (1), 
	       t (1),
	       frame_counter (0),
	       frames_last_second (0)
      {
      }
   };

   Timing   timing;	   //!< frame timing (cf Timing).
   bool	    use_hud;	   //!< true if using Hud.
   bool	    first_time;	   //!< true if first time (or reset).
   
   //! Only constructor.
   DefaultDrawingPolicy (Window& win): ViewingPolicy (win),
				       use_hud (true),
				       first_time (true)
   {
      win.Lock();
      if (!win.IsCreated()){
	 win.Unlock();
	 Error ("Window must be created !","Drawer");
      }
      Viewing::Resize (win);
      win.Unlock();
   }
 
   //! Pre-drawing.
   /*!
    * Pre-drawing is the first step of the drawing cycle. It configures several
    * things in order the window to be ready to accept OpenGL commands. This 
    * also update the frame time counters status.
    */
   void
   PreDraw (Window& win)
   {
      if (first_time){
	 FrameTiming (1);
	 gl_config.SetClientOption (GL_ALL_ATTRIB_BITS,true);
	 first_time = false;
      }
      
      FrameTiming();
      win.GetDrawable().WaitWM(); // we're not really obliged to wait for it.
      win.Act ("drawable");//libs::window::input::drawable);
   }

   //! Post-drawing.
   /*!
    * Post-drawing is the last step of the drawing cycle. It swaps the buffers
    * in order to display what's drawn between PreDraw() and PostDraw().
    */
   void
   PostDraw (Window& win)
   {
      win.Act ("refresh");//libs::window::input::refresh);
      glGetError(); // idem
      win.GetDrawable().WaitGL();
      win.GetDrawable().SwapBuffers();
      win.Act ("refreshed");//libs::window::input::refreshed);
   }

   //! Send true if you want to see and use the Hud.
   inline void 
   SetUseHud (bool val)
   {
      use_hud = val;
   }

   inline bool
   GetUseHud()
   {
      return use_hud;
   }
   
   //! Pre-Hud-drawing.
   /*!
    * This is the first step of the Hud-drawing cycle. The Hud drawing cycle 
    * begins just before PostDraw() is called (so, after the last OpenGL 
    * drawing command).
    */
   void
   PreHudDraw (Window& win)
   {
      Viewing::SetViewmatrix (projection);
      glPushMatrix();
      glLoadIdentity();
      Viewing::SetProjection2D();
      Viewing::SetViewmatrix (modelview);
      glLoadIdentity();
      gl_config.Push();
      win.Act ("hud_drawable");//leg::libs::window::input::hud_drawable);
   }

   //! Post-Hud-drawing.
   /*!
    * This is the last step of the Hud-drawing cycle. This ends the used 2D
    * projection for displaying texts.
    */
   void
   PostHudDraw (Window& win)
   {
      win.Act ("hud_undrawable");//leg::libs::window::input::hud_undrawable);
      gl_config.Pop();
      Viewing::SetViewmatrix (projection);
      glPopMatrix();
   }

   //! Reset first_time to true.
   void
   ResetFirstTime()
   {
      first_time = true;
   }

   private:
   
   void
   FrameTiming (unsigned int reset = 0)
   {
      typedef leg::support::timing::game_time GameTime;
      static GameTime last_time = 0;
      static GameTime time_buff = 0;

      if (reset == 1){
	 last_time = 0;
	 time_buff = 0;
	 timing.frame_counter = 0;
	 timing.clock.SetCurrentTime (0);
	 return;
      }
      
      ++timing.frame_counter;
      
      GameTime cur_time = timing.clock.GetCurrentTime();
      timing.t = cur_time - last_time;
      timing.n = 1/timing.t;

      if ((cur_time - time_buff)>1){
	 timing.frames_last_second = timing.frame_counter;
	 time_buff = cur_time;
	 timing.frame_counter = 0;
	 std::cout << "Frames last second: " << timing.frames_last_second << std::endl;
	 //std::cout << "Framerate status: " << timing.n << " fps." << std::endl;
	 //std::cout << "Time elapsed since last frame: " << timing.t << " sec." << std::endl;
      }

      last_time = cur_time;
   }
};

//! Configurable drawer.
/*!
 * Drawer allows to draw by choosing some template arguments by manipulating
 * them in order to represent the final drawer.
 *
 * tad:
 *    DrawingClass is a class that provides a Draw() member function that codes
 *    all the drawing stuff (3D). It can be whatever class.
 *    HudDrawingClass is almost like DrawingClass but for Hud draws.
 *    Window is the window type (that will be propagated to the drawing policy).
 *    ViewingClass is the class used for the view (default is recommanded, will
 *    be propagated to the viewing policy).
 *    DrawingPolicy is the drawing policy used (can use default).
 *    ViewingPolicy is the viewing policy used (can use default).
 *
 * status:
 *    working;
 *
 * info:
 *    none.
 *
 * @sa DefaultDrawingPolicy,DefaultViewingPolicy,Viewer,Window.
 */
template
<
   class DrawingClass,
   class HudDrawingClass,
   class Window,
   class ViewingClass = Viewer<Perspective,Ortho,Viewport,MatrixMode>,
   template <class,class> class DrawingPolicy = DefaultDrawingPolicy,
   template <class,class> class ViewingPolicy = DefaultViewingPolicy
>
class Drawer:  protected DrawingPolicy<Window,ViewingPolicy<ViewingClass,Window> >,
	       protected DrawingClass,
	       protected HudDrawingClass
{
   protected:

   typedef ViewingPolicy<ViewingClass,Window>	Viewing;
   typedef DrawingPolicy<Window,Viewing>	ParentDrawingPolicy;
   typedef DrawingClass				ParentDrawing;
   typedef HudDrawingClass			ParentHudDrawing;

   //! A reference to the window.
   Window& window;
   
   //! Not useable constructor.
   Drawer();

   public:

   //! Only available constructor.
   Drawer (Window& window):   ParentDrawingPolicy (window),
			      ParentDrawing (),
			      ParentHudDrawing (),
			      window (window)
   {
   }

   ~Drawer()
   {
   }

   //! Draw all.
   void
   Draw()
   {
      ParentDrawingPolicy::PreDraw (window);
      ParentDrawing::Draw ();
      if (ParentDrawingPolicy::use_hud){
	 ParentDrawingPolicy::PreHudDraw (window);
	 ParentHudDrawing::HudDraw ();
	 ParentDrawingPolicy::PostHudDraw (window);
      }
      ParentDrawingPolicy::PostDraw (window);
   }
};

}
}
}
#endif
