#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
from threading import Thread, Lock, Condition
from subprocess import Popen, PIPE
import os
import sys
import Queue

from medialibrary import MediaLibrary


class MediaLibraryThread(Thread):
    """ Controls the Media Library. """

    """
     ATTRIBUTES
    """
    # Media Library command queue
    __commandQueue = Queue.Queue()
    # The media library database
    __mediaLibrary = None
    # Holds the search terms
    __searchTerms = None
    # Indicates to search for audio media
    __searchAudio = None
    # Indicates to search for video media
    __searchVideo = None
    # List of all the songs found from a search
    __songsFound = None

    """
     MEDIA LIBRARY STATES
    """
    # Indicates if the media library is in a running state.
    __blIsRunning = False
    # Indicates if the media library is in a searching state.
    __blIsSearchDone = False

    """
     MEDIA LIBRARY THREAD LOCKS
    """
    # Lock for accessing the command queue.
    __lckCommandQueue = None
    # Condition needed to run a signal command
    __cndtnRunCommand = None

    """
     SIGNAL COMMANDS
    """
    # Indicates a command has not been issued.
    __commandEmpty = hex(0x0)
    # Indicates the media library should quit running.
    __commandQuitRunning = hex(0x1)
    # Indicates the media library should search.
    __commandSearch = hex(0x2)

    def __init__(self):
        """ Create and start the media library thread. """
        Thread.__init__(self, name="Medial Library")
        self.__mediaLibrary = MediaLibrary()

        # Create the locks and conditions
        self.__lckRunningCommand = Lock()
        self.__lckCommandQueue = Lock()
        self.__cndtnRunCommand = Condition(self.__lckRunningCommand)
        self.start()

    def run(self):
        """ Process command queue and issue signals.

         Processes __commandQueue and issue any signals to the media library
           until __commandQuitRunning is placed in the queue and processed.
        """
        self.__blIsRunning = True
        print "Media Library Thread Started"
        while self.__blIsRunning:
            self.__cndtnRunCommand.acquire()
            command = self.__commandEmpty
            while self.__commandQueue.empty():
                self.__cndtnRunCommand.wait()
            self.__lckCommandQueue.acquire()
            command = self.__commandQueue.get(True)
            self.__lckCommandQueue.release()
            print "\nMedia Library Thread Started Processing Command:", command

            # Signal to quit the media library
            if command == self.__commandQuitRunning:
                print "Media Library Thread Running Signal Quit"
                self.__lckCommandQueue.acquire()
                while not self.__commandQueue.empty():
                    self.__commandQueue.get(True)
                self.__lckCommandQueue.release()
                self.__blIsRunning = False
                print "Media Library Thread End Of Signal Quit"
                break

            # Signal to search the media library
            if command == self.__commandSearch:
                print "Media Library Thread Running Signal Search"
                self.__blIsSearchDone = False
                self.__songsFound = self.__mediaLibrary.findSongs( \
                    self.__searchTerms, \
                    self.__searchAudio, \
                    self.__searchVideo)
                self.__blIsSearchDone = True
                print "Media Library Thread End Of Signal Search"

            print "Media Library Thread Finished Processing Command:", command
            self.__cndtnRunCommand.release()
        print "Media Library Thread Ended"

    def __pushCommand(self, command):
        """ Adds a signal command to the command queue

         SignalCommand command: is the command to issue as a signal.
        """
        self.__lckCommandQueue.acquire()
        if self.__blIsRunning:
            print "MediaLibraryThread.__pushCommand(command)", command
            self.__commandQueue.put(command, True)
        self.__lckCommandQueue.release()

    def isRunning(self):
        """ Indicates if the search is in a running state. """
        return self.__blIsRunning

    def quitRunning(self):
        """ Issues the command to quit running. """
        self.__pushCommand(self.__commandQuitRunning)
        self.__cndtnRunCommand.acquire()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def searchPaths(self):
        """ Accessor to the paths to search for new media.

         return list[string]: The search paths for new media.
        """
        self.__cndtnRunCommand.acquire()
        paths = self.__mediaLibrary.searchPaths()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()
        return paths

    def readSettings(self, configuration):
        """ Reads the media library settings.

         Reads in the database path, and the search paths, with each search
           path separated by a ';'
         SafeConfigParser configuration: holds the media library settings.
        """
        self.__cndtnRunCommand.acquire()
        self.__mediaLibrary.readSettings(configuration)
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def writeSettings(self, configuration):
        """ Writes out the media library settings.

         Sets the database path and the search path with each search path
           separated by a ';'
         SafeConfigParser configuration: is the object to save the settings in.
        """
        self.__cndtnRunCommand.acquire()
        self.__mediaLibrary.writeSettings(configuration)
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def supportedMedia(self):
        """ All the supported media file extensions.

         return list[string]: the list of all media file extensions.
        """
        self.__cndtnRunCommand.acquire()
        supportedMedia = self.__mediaLibrary.supportedMedia()
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()
        return supportedMedia

    def buildLibrary(self, paths):
        """ Builds the database of all the audio and video files.

         Creates a new database with the files listed in the search path. If
           a file does not contain an extension of one of the supported media
           formats a message is given to indicate it not being included.
         list[strings] paths: is the paths to the directory to search for all
           audio and video music files.
         return int: the number of songs found
        """
        self.__cndtnRunCommand.acquire()
        songsFound = self.__mediaLibrary.buildLibrary(paths)
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()
        return songsFound

    def findSongs(self, searchTerms, searchAudio, searchVideo):
        """ Finds songs that meet a given criteria.

         Searches through the database and finds songs that meet all of the
           terms the user entered.
         string searchTerms: is the search items in part or in whole of the
           name of a song either by artist or by title. Each term is separated
           by white space. For inclusion in the results, a song must contain
           all the search terms, case insensitive, and there must be terms to
           compare with.
         boolean searchAudio: True if the song must be an audio media.
         boolean searchVideo: True if the song must be a video media.
        """
        self.__cndtnRunCommand.acquire()
        self.__searchTerms = searchTerms
        self.__searchAudio = searchAudio
        self.__searchVideo = searchVideo
        self.__pushCommand(self.__commandSearch)
        self.__cndtnRunCommand.notify()
        self.__cndtnRunCommand.release()

    def searchFinished(self):
        """ Indicates if a search that started has now finished. """
        return self.__blIsSearchDone

    def songsFound(self):
        """ Accessor to the songs found from a search. """
        return self.__songsFound