#!/usr/bin/env python
#******************************************************************************
#**** Copyright (C) 2009  John Schneiderman <JohnMS@member.fsf.org>        ****
#****                                                                      ****
#**** This program is free software: you can redistribute it and/or modify ****
#**** it under the terms of the GNU General Public License as published by ****
#**** the Free Software Foundation, either version 3 of the License, or    ****
#**** (at your option) any later version.                                  ****
#****                                                                      ****
#**** This program is distributed in the hope that it will be useful,      ****
#**** but WITHOUT ANY WARRANTY; without even the implied warranty of       ****
#**** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        ****
#**** GNU General Public License for more details.                         ****
#****                                                                      ****
#**** You should have received a copy of the GNU General Public License    ****
#**** along with this program.  If not, see <http://www.gnu.org/licenses/> ****
#******************************************************************************

"""
 IMPORTS
"""
import wx


class KapDialogue(wx.Dialog):
    """ A dialogue to configure the media library """

    """
     ATTRIBUTES
    """
    # Indicates if the settings should be saved
    blSaveSettings = False
    # Indicates if the a setting has changed
    __blIsSettingsDirty = False
    # Minimum refresh rate for updating any window, in milliseconds
    __MINIMUM_REFRESH_RATE = 0
    # Maximum refresh rate for updating any window, in milliseconds
    __MAXIMUM_REFRESH_RATE = 2500

    """
     Event IDs
    """
    # Indicates that the additional player arguments have changed
    __ID_ARGUMENTS = wx.NewId()
    # Indicates that the player refresh rate has changed
    __ID_PLAYER_REFRESH_RATE = wx.NewId()
    # Indicates that the search result refresh rate has changed
    __ID_SEARCH_REFRESH_RATE = wx.NewId()
    # Indicates that the remove from play-list has changed
    __ID_REMOVE_FROM_PLAY_LIST = wx.NewId()
    # Indicates that the leave in play-list has changed
    __ID_LEAVE_IN_PLAY_LIST = wx.NewId()

    def __init__(self, parent, arguments, playerRefreshRate, \
            searchRefreshRate, removeFromPlayList):
        """ Dialogue constructor.

         wx.Window parent: is the parent window to the dialogue
         string arguments: are the additional arguments to pass to the player.
         int playerRefreshRate: is the refresh rate for the player window.
         int searchRefreshRate: is the refresh rate for the search window.
         bool removeFromPlayList: indicates if a song is to be removed from
           the play-list.
        """
        wx.Dialog.__init__(self, parent, wx.NewId(), "Configure Kap")
        self.__createControls(arguments, playerRefreshRate, \
            searchRefreshRate, removeFromPlayList)
        self.__bindEvents()
        self.__doLayout()

    def __createControls(self, arguments, playerRefreshRate, \
            searchRefreshRate, removeFromPlayList):
        """ Creates all the user controls for the dialogue.

         string arguments: are the additional arguments to pass to the player.
         int playerRefreshRate: is the refresh rate for the player window.
         int searchRefreshRate: is the refresh rate for the search window.
         bool removeFromPlayList: indicates if a song is to be removed from
           the play-list.
        """
        # Argument Controls
        self.__sttcTxtArguments = wx.StaticText(self, wx.NewId(), \
            "Additional Player Arguments:")
        self.__txtCtrlArguments = wx.TextCtrl(self, \
            self.__ID_ARGUMENTS)
        self.__txtCtrlArguments.SetValue(arguments)

        # Player Refresh Rate Controls
        self.__sttcTxtPlayerRefreshRate = wx.StaticText(self, \
            wx.NewId(), "Player Window Refresh Rate (ms):")
        self.__spnCtrlPlayerRefreshRate = wx.SpinCtrl(self, \
            self.__ID_PLAYER_REFRESH_RATE)
        self.__spnCtrlPlayerRefreshRate.SetRange( \
            self.__MINIMUM_REFRESH_RATE, self.__MAXIMUM_REFRESH_RATE)
        self.__spnCtrlPlayerRefreshRate.SetValue(playerRefreshRate)

        # Search Refresh Rate Controls
        self.__sttcTxtSearchRefreshRate = wx.StaticText(self, \
            wx.NewId(), "Search Window Refresh Rate (ms):")
        self.__spnCtrlSearchRefreshRate = wx.SpinCtrl(self, \
            self.__ID_SEARCH_REFRESH_RATE)
        self.__spnCtrlSearchRefreshRate.SetRange( \
            self.__MINIMUM_REFRESH_RATE, self.__MAXIMUM_REFRESH_RATE)
        self.__spnCtrlSearchRefreshRate.SetValue(searchRefreshRate)

        # Remove From Play-List Controls
        self.__sttcTxtRemoveFromPlayList = wx.StaticText(self, \
            wx.NewId(), "Current Playing Song:")
        self.__rdBttnRemoveFromPlayList = wx.RadioButton(self, \
            self.__ID_REMOVE_FROM_PLAY_LIST, \
            "Remove From Play-List")
        self.__rdBttnLeaveInPlayList = wx.RadioButton(self, \
            self.__ID_LEAVE_IN_PLAY_LIST, \
            "Leave In Play-List")
        if removeFromPlayList:
            self.__rdBttnRemoveFromPlayList.SetValue(True)
            self.__rdBttnLeaveInPlayList.SetValue(False)
        else:
            self.__rdBttnRemoveFromPlayList.SetValue(False)
            self.__rdBttnLeaveInPlayList.SetValue(True)

        # Standard Dialogue Buttons
        self.__bttnSaveSettings = wx.Button(self, wx.ID_OK, \
            "Save Settings")
        self.__bttnClose = wx.Button(self, wx.ID_CANCEL, "Close")

    def __bindEvents(self):
        """ Places the controls in a defined layout. """
        wx.EVT_TEXT(self, self.__ID_ARGUMENTS, self.__onMarkSettingsDirty)
        wx.EVT_TEXT(self, self.__ID_PLAYER_REFRESH_RATE, \
            self.__onMarkSettingsDirty)
        wx.EVT_TEXT(self, self.__ID_SEARCH_REFRESH_RATE, \
            self.__onMarkSettingsDirty)
        wx.EVT_RADIOBUTTON(self, self.__ID_REMOVE_FROM_PLAY_LIST, \
            self.__onMarkSettingsDirty)
        wx.EVT_RADIOBUTTON(self, self.__ID_LEAVE_IN_PLAY_LIST, \
            self.__onMarkSettingsDirty)
        wx.EVT_BUTTON(self, wx.ID_OK, self.__onSaveSettings)
        wx.EVT_BUTTON(self, wx.ID_CANCEL, self.__onClose)
        wx.EVT_CLOSE(self, self.__onClose)

    def __doLayout(self):
        """ Places the controls in a defined layout. """

        # Create Player Argument Layout
        self.__bxSzrArguments = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrArguments.Add(self.__sttcTxtArguments, 0, wx.ALL, 3)
        self.__bxSzrArguments.Add(self.__txtCtrlArguments, 1, wx.EXPAND | \
            wx.ALL, 3)

        # Create Player Window Refresh Rate Layout
        self.__bxSzrPlayerRefreshRate = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrPlayerRefreshRate.Add(self.__sttcTxtPlayerRefreshRate, \
            0, wx.ALL, 3)
        self.__bxSzrPlayerRefreshRate.Add(self.__spnCtrlPlayerRefreshRate, \
            0, wx.EXPAND | wx.ALL, 3)

        # Create Search Window Refresh Rate Layout
        self.__bxSzrSearchRefreshRate = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrSearchRefreshRate.Add(self.__sttcTxtSearchRefreshRate, \
            0, wx.ALL, 3)
        self.__bxSzrSearchRefreshRate.Add(self.__spnCtrlSearchRefreshRate, \
            0, wx.EXPAND | wx.ALL, 3)

        # Create Remove From Play-List Layout
        self.__bxSzrRemoveFromPlayList = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrRemoveFromPlayList.Add(self.__sttcTxtRemoveFromPlayList, \
            0, wx.EXPAND | wx.ALL, 3)
        self.__bxSzrRadioList = wx.BoxSizer(wx.HORIZONTAL)
        self.__bxSzrRadioList.AddSpacer(10)
        self.__bxSzrRadioList.Add(self.__rdBttnRemoveFromPlayList, \
            0, wx.EXPAND | wx.ALL, 3)
        self.__bxSzrRadioList.Add(self.__rdBttnLeaveInPlayList, \
            0, wx.EXPAND | wx.ALL, 3)
        self.__bxSzrRemoveFromPlayList.AddSizer(self.__bxSzrRadioList, \
            0, wx.EXPAND | wx.ALL, 3)

        # Create Saving Layout
        self.__stdDlgBttnSzrButtons = wx.StdDialogButtonSizer()
        self.__stdDlgBttnSzrButtons.AddButton(self.__bttnSaveSettings)
        self.__stdDlgBttnSzrButtons.AddButton(self.__bttnClose)
        self.__stdDlgBttnSzrButtons.Realize()

        # Create Main Layout
        self.__bxSzrInterface = wx.BoxSizer(wx.VERTICAL)
        self.__bxSzrInterface.AddSpacer(15)
        self.__bxSzrInterface.Add(self.__bxSzrArguments, 0, wx.EXPAND | \
            wx.ALL, 3)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__bxSzrPlayerRefreshRate, 0, \
            wx.EXPAND | wx.ALL, 3)
        self.__bxSzrInterface.Add(self.__bxSzrSearchRefreshRate, 0, \
            wx.EXPAND | wx.ALL, 3)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__bxSzrRemoveFromPlayList, 0, \
           wx.ALL, 3)
        self.__bxSzrInterface.AddSpacer(5)
        self.__bxSzrInterface.Add(self.__stdDlgBttnSzrButtons, 1, wx.ALL)
        self.SetSizer(self.__bxSzrInterface)

    def __onMarkSettingsDirty(self, event):
        """ Marks the settings as being dirty. """
        self.__blIsSettingsDirty = True

    def __onSaveSettings(self, event):
        """ User wants to save the settings.

         Indicates to save the settings by marking them as dirty.
         wx.Event event: is the event object (Not Used)
        """
        self.blSaveSettings = True
        self.Show(False)
        self.Destroy()

    def __onClose(self, event):
        """ Closes the dialogue.

         Handles the closing of the dialogue. If the settings have been marked
           as being dirty, the user is asked to confirm that they wish to
           discard all of their changes.
         wx.Event event: is the event object (Not Used)
        """
        if self.__blIsSettingsDirty:
            answer = wx.MessageBox("Close and save new settings?", \
                "Settings Changed",  wx.CANCEL | wx.YES_NO | wx.ICON_QUESTION)
            if answer == wx.YES:
                self.blSaveSettings = True
            elif answer == wx.NO:
                self.blSaveSettings = False
            else:
                return
        self.Show(False)
        self.Destroy()

    def playerArguments(self):
        """ Accessor to the additional arguments for the player. """
        return self.__txtCtrlArguments.GetValue()

    def playerRefreshRate(self):
        """ Accessor to the player refresh rate for the player window. """
        return self.__spnCtrlPlayerRefreshRate.GetValue()

    def searchRefreshRate(self):
        """ Accessor to the search refresh rate for the search window. """
        return self.__spnCtrlSearchRefreshRate.GetValue()

    def removeFromPlayList(self):
        """ Accessor to remove from the play-list. """
        if self.__rdBttnRemoveFromPlayList.GetValue():
            return True
        elif self.__rdBttnLeaveInPlayList.GetValue():
            return False
        else:
            raise ValueError("Radio Button Current Playing Song Invalid!")
