// -*- C++ -*-

/* 
 * GChemPaint library
 * application.cc 
 *
 * Copyright (C) 2004-2006 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "application.h"
#include "document.h"
#include "electron.h"
#include "text.h"
#include "plugin.h"
#include "mendeleiev.h"
#include "mesomer.h"
#include "mesomery.h"
#include "mesomery-arrow.h"
#include "reaction.h"
#include "reactant.h"
#include "reaction-step.h"
#include "reaction-arrow.h"
#include "tools.h"
#include "window.h"
#include "zoomdlg.h"
#include <gcu/filechooser.h>
#include <goffice/utils/go-file.h>
#include <bonobo/bonobo-ui-util.h>
#include <sys/stat.h>
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <locale.h>
#include <openbabel/mol.h>
#include <openbabel/obconversion.h>

bool new_ui;

// Objects creation static methods
static Object* CreateMolecule()
{
	return new gcpMolecule();
}

static Object* CreateReaction ()
{
	return new gcpReaction ();
}

static Object* CreateReactant ()
{
	return new gcpReactant ();
}

static Object* CreateReactionStep()
{
	return new gcpReactionStep ();
}

static Object* CreateReactionArrow()
{
	return new gcpReactionArrow (NULL);
}

static Object* CreateMesomery ()
{
	return new gcpMesomery ();
}

static Object* CreateMesomeryArrow()
{
	return new gcpMesomeryArrow(NULL);
}

static Object* CreateMesomer ()
{
	return new gcpMesomer ();
}

static Object* CreateText()
{
	return new gcpText();
}

static Object* CreateFragment()
{
	return new gcpFragment();
}

list<gcpApplication*> Apps;
bool	gcpApplication::m_bInit = false;
bool	gcpApplication::m_Have_Ghemical = false;
bool	gcpApplication::m_Have_InChI = false;
map<string, string> gcpApplication::m_SupportedPixbufFormats;

#warning "the following lines should be removed for stable releases"
#undef PACKAGE
#define PACKAGE "gchempaint-unstable" 

gcpApplication::gcpApplication (): Application ("GChemPaint", DATADIR, PACKAGE, "gchempaint")
{
	Apps.push_back(this);
	m_CurZ = 6;
	m_pActiveDoc = NULL;
	m_pActiveTool = NULL;
	m_pNode = NULL;
	m_NumWindow = 1;
	FontName = g_strdup("Bitstream Vera Serif 12");
	CurDir = g_get_current_dir ();
	GConfClient* cli = gconf_client_get_default ();
	if (cli) {
		const char *value;
/*		GConfEntry* entry = gconf_client_get_entry (cli, "/desktop/gnome/applications/help_viewer/exec", NULL, true, NULL);
		if (entry) {
			value = gconf_value_get_string (gconf_entry_get_value (entry));
			if (value) HelpBrowser = value;
		}*/
		GConfEntry* entry = gconf_client_get_entry (cli, "/desktop/gnome/applications/browser/exec", NULL, true, NULL);
		if (entry) {
			value = gconf_value_get_string (gconf_entry_get_value (entry));
			if (value) WebBrowser = value;
		}
		entry = gconf_client_get_entry (cli, "/desktop/gnome/url-handlers/mailto/command", NULL, true, NULL);
		if (entry) {
			value = gconf_value_get_string (gconf_entry_get_value (entry));
			if (value) {
				MailAgent = value;
				int i = MailAgent.find (" %s");
				if (i)
					MailAgent.erase (i, MailAgent.size ());
			}
		}
	}

	// search help file
/*	string lang = getenv ("LANG");
	HelpFilename = string (DATADIR"/gnome/help/") + string (PACKAGE"/") + lang + "/"PACKAGE".xml";
	struct stat buf;
	int err = stat(HelpFilename.c_str (), &buf);
	if (err) {
		if (lang.size () > 2) {
			lang.erase (2, lang.size());
			HelpFilename = string (DATADIR"/gnome/help/"PACKAGE"/") + lang + "/"PACKAGE".xml";
			err = stat(HelpFilename.c_str (), &buf);
		}
		if (err) {
			HelpFilename = DATADIR"/gnome/help/"PACKAGE"/C/"PACKAGE".xml";
			err = stat(HelpFilename.c_str (), &buf);
		}
	}
	if (err) {
		g_warning("Help file not found");
		HelpFilename = "";
	}*/

	if (!m_bInit) {
		// Check for programs
		char *result = NULL, *errors = NULL;
		// check for ghemical
		m_Have_Ghemical = (g_spawn_command_line_sync ("which ghemical", &result, &errors, NULL, NULL)
			&& result && strlen (result));
		if (result) {
			g_free (result);
			result = NULL;
		}
		if (errors) {
			g_free (errors);
			errors = NULL;
		}
		OBConversion Conv;
		m_Have_InChI = Conv.FindFormat ("inchi") != NULL || 
			(g_spawn_command_line_sync ("which main_inchi", &result, &errors, NULL, NULL)
			&& result && strlen (result));
		if (result)
			g_free (result);
		if (errors) {
			g_free (errors);
			errors = NULL;
		}
		// check supported pixbuf formats
		GSList *formats = gdk_pixbuf_get_formats ();
		GSList *l = formats;
		GdkPixbufFormat *format;
		char **mimes, **mime;
		while (l) {
			format = (GdkPixbufFormat*) l->data;
			if (gdk_pixbuf_format_is_writable (format)) {
				mimes = gdk_pixbuf_format_get_mime_types (format);
				mime = mimes;
				while (*mime)
					m_SupportedPixbufFormats[*mime++] = gdk_pixbuf_format_get_name (format);
				g_strfreev (mimes);
			}
			l = l->next;
		}
		g_slist_free (formats);

		gcpText::InitTags(FontName);
		// Initialize types
		Object::AddType ("molecule", CreateMolecule, MoleculeType);
		Object::AddType ("reaction", CreateReaction, ReactionType);
		Object::SetCreationLabel (ReactionType, _("Create a new reaction"));
		ReactionStepType = Object::AddType ("reaction-step", CreateReactionStep);
		Object::AddType ("reactant", CreateReactant, ReactantType);
		Object::AddType("reaction-arrow", CreateReactionArrow, ReactionArrowType);
		MesomerType = Object::AddType ("mesomer", CreateMesomer);
		Object::AddType ("mesomery", CreateMesomery, MesomeryType);
		Object::SetCreationLabel (MesomeryType, _("Create a new mesomery relationship"));
		Object::AddType("mesomery-arrow", CreateMesomeryArrow, MesomeryArrowType);
		Object::AddType("text", CreateText, TextType);
		Object::AddType("fragment", CreateFragment, FragmentType);
		ElectronType = Object::AddType ("electron", NULL);
		//Add rules
		Object::AddRule ("reaction", RuleMustContain, "reaction-step");
		Object::AddRule ("reaction-step", RuleMustContain, "reactant");
		Object::AddRule ("reactant", RuleMustBeIn, "reaction-step");
		Object::AddRule ("reaction-step", RuleMustBeIn, "reaction");
		Object::AddRule ("reaction", RuleMustContain, "reaction-arrow");
		Object::AddRule ("reaction-arrow", RuleMustBeIn, "reaction");
		Object::AddRule ("reactant", RuleMayContain, "molecule");
		Object::AddRule ("mesomer", RuleMustContain, "molecule");
		Object::AddRule ("mesomer", RuleMustBeIn, "mesomery");
		Object::AddRule ("mesomery", RuleMustContain, "mesomer");
		Object::AddRule ("mesomery", RuleMustContain, "mesomery-arrow");
		Object::AddRule ("mesomery-arrow", RuleMustBeIn, "mesomery");

		// Create global signal ids
		OnChangedSignal = Object::CreateNewSignalId ();
		OnDeleteSignal = Object::CreateNewSignalId ();

		gcpPlugin::LoadPlugins();
		m_bInit = true;
	}
	RadioActions = NULL;
	m_entries = 0;
	IconFactory = gtk_icon_factory_new ();
	set<gcpPlugin*>::iterator i = Plugins.begin (), end = Plugins.end ();
	while (i != end) (*i++)->Populate (this);
	gtk_icon_factory_add_default (IconFactory);
	g_object_unref (G_OBJECT (IconFactory));
	XmlDoc = xmlNewDoc ((xmlChar const*) "1.0");
	visible_windows = 0;
	m_idle = 0;
	m_SupportedMimeTypes.push_back ("application/x-gchempaint");
	m_SupportedMimeTypes.push_back ("chemical/x-chemdraw");
	m_SupportedMimeTypes.push_back ("chemical/x-cml");
	m_SupportedMimeTypes.push_back ("chemical/x-mdl-molfile");
	m_SupportedMimeTypes.push_back ("chemical/x-pdb");
}

gcpApplication::~gcpApplication()
{
	map<string, gcpTool*>::iterator tool = Tools.begin(), endtool = Tools.end();
	for (; tool!= endtool; tool++)
		delete (*tool).second;
	Tools.clear ();
	Apps.remove (this);
	g_free (FontName);
	if (m_pNode)
		bonobo_ui_node_free (m_pNode);
	if (XmlDoc)
		xmlFreeDoc (XmlDoc);
	m_SupportedMimeTypes.clear ();
}

void gcpApplication::ActivateMenu(const string& menuname, bool activate)
{
}

void gcpApplication::ActivateToolItem(const string& itemname, bool activate)
{
}

void gcpApplication::ActivateTool(const string& toolname, bool activate)
{
	if (Tools[toolname]) {
		if (activate) {
			if (m_pActiveTool != Tools[toolname]) {
				if (m_pActiveTool)
					m_pActiveTool->Activate (false);
				m_pActiveTool = Tools[toolname];
				m_pActiveTool->Activate (true);
				GtkToggleToolButton* button = (GtkToggleToolButton*) ToolItems[toolname];
				if (button && !gtk_toggle_tool_button_get_active (button))
					gtk_toggle_tool_button_set_active (button, true);
			}
		} else {
			if (m_pActiveTool == Tools[toolname])
				m_pActiveTool = NULL;
			Tools[toolname]->Activate (false);
		}
	}
}

void gcpApplication::ClearStatus()
{
}

void gcpApplication::SetStatusText(const char* text)
{
}

void gcpApplication::OnSaveAs()
{
	FileChooser (this, true, m_SupportedMimeTypes, m_pActiveDoc);
}

void gcpApplication::AddUI (const char* filename)
{
	BonoboUINode *Node, *Parent = bonobo_ui_node_from_file (filename);
	bonobo_ui_util_translate_ui  (Parent);
	if (!m_pNode)
		m_pNode = Parent;
	else 
	{
		BonoboUINode *Child, *Next;
		if (!strcmp (bonobo_ui_node_get_name (Parent), "Root"))
			Node = bonobo_ui_node_children (Parent);
		else
			Node = NULL;
		while (Node)
		{
			const char* name = bonobo_ui_node_get_name (Node);
			char* id = bonobo_ui_node_get_attr (Node, "name");
			Child = bonobo_ui_node_children (m_pNode);
			while (Child)
			{
				char *child_id = bonobo_ui_node_get_attr (Child, "name");
				if (!strcmp(name, bonobo_ui_node_get_name (Child))
					&& !strcmp(id, child_id))
				{
					MergeNodes (Child, Node);
					break;
				}
				bonobo_ui_node_free_string (child_id);
				Child = bonobo_ui_node_next (Child);
			}
			bonobo_ui_node_free_string (id);
			Next = bonobo_ui_node_next (Node);
			if (!Child) {
				bonobo_ui_node_unlink (Node);
				bonobo_ui_node_add_child (m_pNode, Node);
			}
			Node = Next;
		}
		bonobo_ui_node_free (Parent);
	}
}

bool gcpApplication::FileProcess(const gchar* filename, const gchar* mime_type, bool bSave, GtkWindow *window, Document *Doc)
{
	const gchar* ext;
	gcpDocument *pDoc = static_cast<gcpDocument*> (Doc);
	if (!filename || !strlen(filename) || filename[strlen(filename) - 1] == '/')
	{
		GtkWidget* message = gtk_message_dialog_new (window, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
															_("Please enter a file name,\nnot a directory"));
		gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
		gtk_dialog_run (GTK_DIALOG (message));
		gtk_widget_destroy (message);
		return true;
	}
	if (!mime_type)
		mime_type = "application/x-gchempaint";
	int filetype = (mime_type)? 0: gcp_ft_Normal;
	int n = strlen (filename), i = n - 1;
	while ((i > 0) && (filename[i] != '.') && (filename[i] != '/')) i--;
	if (filename[i] == '/') i = 0;
	ext = (i > 0)? filename + i: NULL;
	if (!filetype) {
		// Check mime type
		if (!strcmp (mime_type, "application/x-gchempaint"))
			filetype = gcp_ft_Normal;
		else if (!strcmp (mime_type, "chemical/x-cml"))
			filetype = gcp_ft_CML;
		else if (!strcmp (mime_type, "chemical/x-mdl-molfile"))
			filetype = gcp_ft_SDF;
		else if (!strcmp (mime_type, "chemical/x-pdb"))
			filetype = gcp_ft_PDB;
		else if (!strcmp (mime_type, "chemical/x-chemdraw"))
			filetype = gcp_ft_ChemDraw;
		else {
			GtkWidget* message = gtk_message_dialog_new (window, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
														_("Sorry, format not supported!"));
			gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
			return true;
		}
	}
	if (!ext)
		switch (filetype)
		{
			case gcp_ft_Normal: ext = ".gchempaint"; break;
			case gcp_ft_CML: ext = ".cml"; break;
			case gcp_ft_SDF:
				if (!strcmp(filename + n - 3, "sdf")) ext = ".sdf";
				else if (!strcmp(filename + n - 2, "sd")) ext = ".sd";
				else if (bSave) ext = ".mol";
				break;
			case gcp_ft_PDB:
				if (!strcmp(filename + n - 3, "ent")) ext = ".ent";
				else if (bSave) ext = ".pdb";
				break;
			case gcp_ft_ChemDraw: ext = ".chm"; break;
			default:
				ext = NULL;
				break;
		}
	gchar* filename2;
	bool err;
	GnomeVFSURI *uri;
	if (bSave)
	{
		if (ext)
		{
			int i = strlen(filename) - strlen(ext);
			if ((i > 0) && (!strcmp(filename +i, ext)))
				filename2 = g_strdup(filename);
			else
				filename2 = g_strdup_printf("%s%s", filename, ext);
		}
		else filename2 = g_strdup(filename);
		uri = gnome_vfs_uri_new (filename2);
		err = gnome_vfs_uri_exists (uri);
		gnome_vfs_uri_unref (uri);
		gint result = GTK_RESPONSE_YES;
		if (err)
		{
			gchar * message = g_strdup_printf(_("File %s\nexists, overwrite?"), filename2);
			GtkDialog* Box = GTK_DIALOG(gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, message));
			gtk_window_set_icon_name (GTK_WINDOW (Box), "gchempaint");
			result = gtk_dialog_run(Box);
			gtk_widget_destroy(GTK_WIDGET(Box));
			g_free(message);
		}
		if (result == GTK_RESPONSE_YES)
		{
			// destroy the old file
			gnome_vfs_unlink (filename2);
			pDoc->SetType (filetype);
			if (filetype == gcp_ft_Normal)
				SaveGcp (filename2, pDoc);
			else
				SaveWithBabel (filename2, ext, pDoc, filetype);
		}
	}
	else //loading
	{
		uri = gnome_vfs_uri_new (filename);
		err = !gnome_vfs_uri_exists (uri);
		gnome_vfs_uri_unref (uri);
		if (err)
		{
			if (!ext)
				switch (filetype)
				{
					case gcp_ft_SDF:
						filename2 = g_strdup_printf("%s.%s",filename, "mol");
						uri = gnome_vfs_uri_new (filename2);
						err = !gnome_vfs_uri_exists (uri);
						gnome_vfs_uri_unref (uri);
						if (!err)
							break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "sdf");
						uri = gnome_vfs_uri_new (filename2);
						err = !gnome_vfs_uri_exists (uri);
						gnome_vfs_uri_unref (uri);
						if (!err)
							break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "sd");
						uri = gnome_vfs_uri_new (filename2);
						err = !gnome_vfs_uri_exists (uri);
						gnome_vfs_uri_unref (uri);
						if (!err)
							break;
						g_free(filename2);
						filename2 = g_strdup(filename);
						break;
					case gcp_ft_PDB:
						filename2 = g_strdup_printf("%s.%s",filename, "pdb");
						uri = gnome_vfs_uri_new (filename2);
						err = !gnome_vfs_uri_exists (uri);
						gnome_vfs_uri_unref (uri);
						if (!err)
							break;
						g_free(filename2);
						filename2 = g_strdup_printf("%s.%s",filename, "ent");
						uri = gnome_vfs_uri_new (filename2);
						err = !gnome_vfs_uri_exists (uri);
						gnome_vfs_uri_unref (uri);
						if (!err)
							break;
						g_free(filename2);
						filename2 = g_strdup(filename);
						break;
					default:
						filename2 = g_strdup(filename);
				}
			else filename2 = g_strdup_printf("%s%s",filename, ext);
			uri = gnome_vfs_uri_new (filename2);
			err = !gnome_vfs_uri_exists (uri);
			gnome_vfs_uri_unref (uri);
			if (err)
			{
				g_free(filename2);
				filename2 = g_strdup(filename);
			}
		}
		else filename2 = g_strdup(filename);
		if (filetype == gcp_ft_Normal)
			OpenGcp (filename2, pDoc);
		else
			OpenWithBabel (filename2, ext, pDoc, filetype);
	}
	g_free(filename2);
	return false;
}

void gcpApplication::SaveWithBabel(const gchar* filename, const gchar* ext, gcpDocument* pDoc, int Type)
{
	pDoc->SetFileName(filename, ext);
	pDoc->SetType(Type);
	pDoc->Save();
}

void gcpApplication::OpenWithBabel(const gchar* filename, const gchar* ext, gcpDocument* pDoc, int Type)
{
	char *old_num_locale;
	bool bNew = (pDoc == NULL), local;
	GnomeVFSFileInfo *info = NULL;
	bool result;
	try
	{
		if (!filename) throw (int) 0;
		info = gnome_vfs_file_info_new ();
		gnome_vfs_get_file_info (filename, info, GNOME_VFS_FILE_INFO_DEFAULT);
		local = GNOME_VFS_FILE_INFO_LOCAL (info);
		gnome_vfs_file_info_unref (info);
		if (bNew)
			pDoc = new gcpDocument (this, true);
		if (local) {
			ifstream ifs;
			GnomeVFSURI *uri = gnome_vfs_uri_new (filename);
			ifs.open (gnome_vfs_uri_get_path (uri));
			gnome_vfs_uri_unref (uri);
			if (!ifs) throw (int) 1;
			old_num_locale = g_strdup (setlocale (LC_NUMERIC, NULL));
			setlocale(LC_NUMERIC, "C");
			OBMol Mol;
			OBConversion Conv;
			OBFormat* pInFormat = Conv.FormatFromExt (filename);
			Conv.SetInAndOutFormats (pInFormat, pInFormat);
			Conv.Read (&Mol, &ifs);
			result = pDoc->ImportOB(Mol);
			setlocale (LC_NUMERIC, old_num_locale);
			g_free (old_num_locale);
			ifs.close ();
		} else {
			char *buf;
			int size;
			if (gnome_vfs_read_entire_file (filename, &size, &buf) != GNOME_VFS_OK)
				throw 1;
			istringstream iss (buf);
			old_num_locale = g_strdup (setlocale (LC_NUMERIC, NULL));
			setlocale(LC_NUMERIC, "C");
			OBMol Mol;
			OBConversion Conv;
			OBFormat* pInFormat = Conv.FormatFromExt (filename);
			Conv.SetInAndOutFormats (pInFormat, pInFormat);
			Conv.Read (&Mol, &iss);
			result = pDoc->ImportOB(Mol);
			setlocale (LC_NUMERIC, old_num_locale);
			setlocale (LC_NUMERIC, old_num_locale);
			g_free (old_num_locale);
			g_free (buf);
		}
		if (!result)
		{
			if (bNew) delete pDoc;
			throw (int) 2;
		}
		pDoc->SetFileName(filename, ext);
		pDoc->SetType(Type);
		if (bNew) AddDocument(pDoc);
	}
	catch (int num)
	{
		gchar *mess = NULL;
		GtkWidget* message;
		switch (num)
		{
		case 0:
			mess = g_strdup_printf(_("No filename"));
			break;
		case 1:
			mess = g_strdup_printf(_("Could not open file\n%s"), filename);
			break;
		case 2:
			mess = g_strdup_printf(_("%s: parse error."), filename);
			break;
		default:
			throw (num); //this should not occur
		}
		message = gtk_message_dialog_new(NULL, (GtkDialogFlags) 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, mess);
		gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_widget_show(message);
		g_free(mess);
	}
}

void gcpApplication::SaveGcp(const gchar* filename, gcpDocument* pDoc)
{
	pDoc->SetFileName(filename, ".gchempaint");
	pDoc->Save();
}

static int	cb_vfs_to_xml (GnomeVFSHandle *handle, char* buf, int nb)
{
	GnomeVFSFileSize ndone;
	return  (gnome_vfs_read (handle, buf, nb, &ndone) == GNOME_VFS_OK)? (int) ndone: -1;
}

void gcpApplication::OpenGcp(const gchar* filename, gcpDocument* pDoc)
{
	xmlDocPtr xml = NULL;
	char *old_num_locale, *old_time_locale;
	GnomeVFSFileInfo *info = NULL;
	try
	{
		if (!filename)
			throw (int) 0;

		// try opening with write access to see if it is readonly
		// use xmlReadIO for non local files.
		info = gnome_vfs_file_info_new ();
		gnome_vfs_get_file_info (filename, info, GNOME_VFS_FILE_INFO_DEFAULT);

		if (GNOME_VFS_FILE_INFO_LOCAL (info)) {
			if (!(xml = xmlParseFile(filename)))
				throw (int) 1;
		} else {
			GnomeVFSHandle *handle;
			GnomeVFSResult result = gnome_vfs_open (&handle, filename, GNOME_VFS_OPEN_READ);
			if (result != GNOME_VFS_OK)
				throw 1;
			if (!(xml = xmlReadIO ((xmlInputReadCallback) cb_vfs_to_xml, 
					 (xmlInputCloseCallback) gnome_vfs_close, handle, filename, NULL, 0)))
				throw 1;
		}
		if (xml->children == NULL) throw (int) 2;
		if (strcmp((char*)xml->children->name, "chemistry")) throw (int) 3;	//FIXME: that could change when a dtd is available
		old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
		setlocale(LC_NUMERIC, "C");
		old_time_locale = g_strdup(setlocale(LC_TIME, NULL));
		setlocale(LC_TIME, "C");
		if (!pDoc) {
			OnFileNew ();
			pDoc = m_pActiveDoc;
		}
		pDoc->SetFileName(filename, ".gchempaint");
		bool result = pDoc->Load(xml->children);
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		setlocale(LC_TIME, old_time_locale);
		g_free(old_time_locale);
		if (!result)
		{
			OnFileClose ();
			throw (int) 4;
		}
		if (!(info->permissions & (GNOME_VFS_PERM_USER_WRITE | GNOME_VFS_PERM_GROUP_WRITE)))
			pDoc->SetReadOnly (true);
		gnome_vfs_file_info_unref (info);
		xmlFreeDoc(xml);
	}
	catch (int num)
	{
		if (info)
			gnome_vfs_file_info_unref (info);
		if (num > 1) xmlFreeDoc(xml);
		gchar *mess = NULL;
		GtkWidget* message;
		switch (num)
		{
		case 0:
			mess = g_strdup_printf(_("No filename"));
			break;
		case 1:
			mess = g_strdup_printf(_("Could not load file\n%s"), filename);
			break;
		case 2:
			mess = g_strdup_printf(_("%s: invalid xml file.\nTree is empty?"), filename);
			break;
		case 3:
			mess = g_strdup_printf(_("%s: invalid file format."), filename);
			break;
		case 4:
			mess = g_strdup_printf(_("%s: parse error."), filename);
			break;
		default:
			throw (num); //this should not occur
		}
		message = gtk_message_dialog_new(NULL, (GtkDialogFlags) 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, mess);
		gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_widget_show(message);
		g_free(mess);
	}
}

void gcpApplication::MergeNodes(BonoboUINode* Dest, BonoboUINode* Source)
{
	BonoboUINode *next, *child = bonobo_ui_node_children (Source);
	while (child)
	{
		next = bonobo_ui_node_next (child);
		bonobo_ui_node_unlink (child);
		bonobo_ui_node_add_child (Dest, child);
		child = next;
	}
}

void gcpApplication::SetFontName(char* fontname)
{
	g_free (FontName);
	FontName = g_strdup (fontname);
}

void gcpApplication::InitTools()
{
	map<string, gcpTool*>::iterator i = Tools.begin (), end = Tools.end ();
	for (; i != end; i++)
		if ((*i).second)
			(*i).second->Activate ((*i).first == "Select");
}

void gcpApplication::OnBug ()
{
	if (!WebBrowser.size ())
		return;
	char *argv[3] = {NULL, PACKAGE_BUGREPORT, NULL};
	argv[0] = (char*) WebBrowser.c_str();
	g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		NULL, NULL, NULL, NULL);
}

void gcpApplication::ShowURI (string& uri)
{
	if (!WebBrowser.size ())
		return;
	char *argv[3] = {NULL, NULL, NULL};
	argv[0] = (char*) WebBrowser.c_str();
	argv[1] = (char*) uri.c_str ();
	g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		NULL, NULL, NULL, NULL);
}

void gcpApplication::OnWeb ()
{
	if (!WebBrowser.size ())
		return;
	char *argv[3] = {NULL, "http://www.nongnu.org/gchempaint", NULL};
	argv[0] = (char*) WebBrowser.c_str();
	g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		NULL, NULL, NULL, NULL);
}

void gcpApplication::OnMail ()
{
	if (!MailAgent.size ())
		return;
	char *argv[3] = {NULL, "mailto:gchempaint-main@nongnu.org", NULL};
	argv[0] = (char*) MailAgent.c_str();
	g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		NULL, NULL, NULL, NULL);
}

void gcpApplication::OnShowElements ()
{
	gcpMendeleievDlg* dlg = (gcpMendeleievDlg*) GetDialog("Mendeleiev");
	if (!dlg)
	{
		dlg = new gcpMendeleievDlg (this, GetCurZ());
		ToggleMenu ("Mendeleiev", true);
	}
	else gdk_window_raise (GTK_WIDGET(dlg->GetWindow())->window); 
}

void gcpApplication::OnSaveAsImage ()
{
	char* filename;
	GtkFileChooser *dialog = (GtkFileChooser*) gtk_file_chooser_dialog_new (
															_("Save as image"),
															GetWindow(),
															GTK_FILE_CHOOSER_ACTION_SAVE,
															GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
															GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
															NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	GtkFileFilter* filter = gtk_file_filter_new ();
	map<string, string>::iterator i, end = m_SupportedPixbufFormats.end ();
	for (i = m_SupportedPixbufFormats.begin (); i != end; i++)
		gtk_file_filter_add_mime_type (filter, (*i).first.c_str ());
	gtk_file_filter_add_mime_type (filter, "image/svg+xml");
	gtk_file_filter_add_mime_type (filter, "image/x-eps");
	gtk_file_filter_add_mime_type (filter, "image/x-xfig");
	gtk_file_chooser_set_filter (dialog, filter);
	while (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		filename = gtk_file_chooser_get_filename (dialog);
		if (!filename || !strlen(filename) || filename[strlen(filename) - 1] == '/') {
			GtkWidget* message = gtk_message_dialog_new (GTK_WINDOW (dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
																_("Please enter a file name,\nnot a directory"));
			gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
			continue;
		}
		struct stat buf;
		gint err;
		err = stat(filename, &buf);
		gint result = GTK_RESPONSE_YES;
		if (!err)
		{
			gchar * message = g_strdup_printf(_("File %s\nexists, overwrite?"), filename);
			GtkDialog* Box = GTK_DIALOG(gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, message));
			gtk_window_set_icon_name (GTK_WINDOW (Box), "gchempaint");
			result = gtk_dialog_run(Box);
			gtk_widget_destroy(GTK_WIDGET(Box));
			g_free(message);
			if (result != GTK_RESPONSE_YES)
				continue;
		}
		const char* mime_type = gnome_vfs_mime_type_from_name (filename);
		if (m_SupportedPixbufFormats.find (mime_type) != m_SupportedPixbufFormats.end())
			m_pActiveDoc->ExportImage (filename, m_SupportedPixbufFormats[mime_type].c_str ());
		else if (!strcmp (mime_type, "image/x-eps"))
				m_pActiveDoc->ExportImage (filename, "eps");
		else if (!strcmp (mime_type, "image/svg+xml")) {
			m_pActiveDoc->ExportImage (filename, "svg");
		} else {
			GtkWidget* message = gtk_message_dialog_new (GTK_WINDOW (dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
																_("Sorry, format not supported!"));
			gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
			gtk_dialog_run (GTK_DIALOG (message));
			gtk_widget_destroy (message);
			continue;
		}
		break;//replace with sensible code
	}
	gtk_widget_destroy (GTK_WIDGET (dialog));
	return;	
}

void gcpApplication::SetCurDir (char const* dir)
{
	if (CurDir)
		g_free (CurDir);
	CurDir = g_strdup (dir);
}

void gcpApplication::Zoom (double zoom)
{
	gcpView *pView = m_pActiveDoc->GetView ();
	// authorized zooms: 20% to 800% all other values will open the zoom dialog.
	if (zoom >= 0.2 && zoom <= 8.)
		pView->Zoom (zoom);
	else {
		Dialog *pDialog = GetDialog ("Zoom");
		if (pDialog)
			gtk_window_present (pDialog->GetWindow ()); 
		else
			new gcpZoomDlg (this);
	}
}

static void on_tool_changed (GtkAction *action, GtkAction *current, gcpApplication* App)
{
	App->OnToolChanged (current);
}

void gcpApplication::OnToolChanged (GtkAction *current)
{
	if (m_pActiveTool)
		m_pActiveTool->Activate(false);
	m_pActiveTool = Tools[gtk_action_get_name (current)];
	gcpTools *Tools = dynamic_cast<gcpTools*> (Dialogs["Tools"]);
	if (Tools)
		Tools->OnSelectTool (m_pActiveTool);		
	if (m_pActiveTool)
		m_pActiveTool->Activate(true);
}

void gcpApplication::BuildTools ()
{
	gcpTools *ToolsBox = new gcpTools (this);
	map<int, string>::iterator i, iend = ToolbarNames.end ();
	list<char const*>::iterator j, jend = UiDescs.end ();
	string s;
	GError *error = NULL;
	GtkUIManager *ToolsManager = gtk_ui_manager_new ();
	ToolsBox->SetUIManager (ToolsManager);
	GtkActionGroup *action_group = gtk_action_group_new ("Tools");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_radio_actions (action_group, RadioActions, m_entries, 0, G_CALLBACK (on_tool_changed), this);
	gtk_ui_manager_insert_action_group (ToolsManager, action_group, 0);
	for (j = UiDescs.begin (); j != jend; j++)
		if (!gtk_ui_manager_add_ui_from_string (ToolsManager, *j, -1, &error)) {
			g_message ("building user interface failed: %s", error->message);
			g_error_free (error);
			exit (EXIT_FAILURE);
		}
	for (i = ToolbarNames.begin (); i != iend; i++) {
		s = "ui/";
		s += (*i).second;
		ToolsBox->AddToolbar (s);
	}
	g_object_unref (ToolsManager);
	m_pActiveTool = Tools["Select"];
	if (m_pActiveTool)
		m_pActiveTool->Activate(true);
	ToolsBox->OnSelectTool (m_pActiveTool);		
	gcpMendeleievDlg::SetElement (this, m_CurZ);
}

void gcpApplication::ShowTools (bool visible)
{
	gcpTools *Tools = dynamic_cast<gcpTools*> (Dialogs["Tools"]);
	if (!Tools) {
		if (visible)
			BuildTools ();
	} else
	Tools->Show (visible);
}

void gcpApplication::AddActions (GtkRadioActionEntry const *entries, int nb, char const *ui_description, gcpIconDesc const *icons)
{
	static int cur_entry = 1;
	if (nb > 0) {
		if (m_entries)
			RadioActions = g_renew (GtkRadioActionEntry, RadioActions, m_entries + nb);
		else
			RadioActions = g_new (GtkRadioActionEntry, nb);
		memcpy (RadioActions + m_entries, entries, nb * sizeof (GtkRadioActionEntry));
		for (int i = 0; i < nb; i++)
			if (strcmp (RadioActions[i + m_entries].name, "Select"))
				RadioActions[i + m_entries].value = cur_entry++;
			else
				RadioActions[i + m_entries].value = 0;
		m_entries += nb;
	}
	if (ui_description)
		UiDescs.push_back (ui_description);
	if (icons) {
		GtkIconSet *set;
		GtkIconSource *src;
		while (icons->name) {
			set = gtk_icon_set_new ();
			src = gtk_icon_source_new ();
		
			gtk_icon_source_set_size_wildcarded (src, TRUE);
			gtk_icon_source_set_pixbuf (src,
				gdk_pixbuf_new_from_inline (-1, icons->data_24, FALSE, NULL));
			gtk_icon_set_add_source (set, src);	/* copies the src */
		
			gtk_icon_factory_add (IconFactory, icons->name, set);	/* keeps reference to set */
			gtk_icon_set_unref (set);
			gtk_icon_source_free (src);
			icons++;
		}
	}
}

void gcpApplication::RegisterToolbar (char const *name, int index)
{
	if (ToolbarNames[index] == "")
		ToolbarNames[index] = name;
}

void gcpApplication::AddWindow (gcpWindow *window)
{
	m_Windows.insert (window);
	NotifyIconification (false);
}

void gcpApplication::DeleteWindow (gcpWindow *window)
{
	m_Windows.erase (window);
	ShowTools (false);
}

void gcpApplication::NotifyIconification (bool iconified)
{
	if (iconified) {
		ShowTools (false);
	}
}

static bool on_wait_focus (gcpApplication *App)
{
	App->CheckFocus ();
	return false;
}

void gcpApplication::CheckFocus ()
{
	ShowTools (focused || gtk_grab_get_current ());
	m_idle = 0;
}

void gcpApplication::NotifyFocus (bool has_focus, gcpWindow *window)
{
	if (window) {
		m_pActiveWin = window;
		m_pActiveDoc = window->GetDocument ();
	}
	focused = has_focus;
	if (!m_idle)
		g_idle_add ((GSourceFunc) on_wait_focus, this);
}

void gcpApplication::CloseAll ()
{
	while (!m_Windows.empty ())
		(*m_Windows.begin ())->Destroy ();
}
