// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// fftwTest1: basic test of FFTW wrapper.
//-----------------------------------------------------------------------------

// Bring in Pooma array machinery.

#include "Pooma/Arrays.h"
#include "Utilities/Tester.h"
#include "Transform/WrapFFTW.h"

static bool OK = true;

#if POOMA_FFTW

inline
void test1(const Array<2, complex<double>, BrickView > &a,
	   Pooma::Tester &tester)
{
  Interval<2> dom = a.domain();

  int length  = dom[0].size();
  int lot     = dom[1].size();
  int istride = a.engine().strides()[0];
  int idist   = a.engine().strides()[1];

  double pi = 3.1415926535897932;

  a = sin(2 * pi * (1 + iota(dom).comp(1) ) * iota(dom).comp(0) / length);

  FftwCC fft(1);  // forward fft

  fft.initialize(Loc<2>(length, lot), Loc<2>(istride, idist));

  fft.apply(&a(0,0));

  fft.finalize();

  complex<double> result = double(length) / complex<double>(0.0, 2.0);

  Array<2, complex<double>, Brick> compare(length, lot);

  compare = complex<double>(0.0, 0.0);

  Pooma::blockAndEvaluate();

  int col;
  for (col = 0; col < lot; ++col)
  {
    compare(col + 1, col) = result;
    compare(length - col - 1, col) = -result;
  }

  double diff = sum(abs(a - compare));

  tester.out() << "difference:" << diff << std::endl;

  tester.check("checking result of transform of sin",
	       diff < 0.00001);

  tester.out() << a << std::endl;
}

#endif // POOMA_FFTW

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

#if POOMA_FFTW

  Interval<1> I1(20);
  Interval<1> Is(10,14);
  Interval<3> I3(I1, I1, I1);

  Array<3, complex<double>, Brick> a(I3);

  test1(a(1, I1, Is), tester);

#else // POOMA_FFTW

  tester.out() << "Warning: FFTW is not enabled, so this" << std::endl
	       << "test is empty.  To enable FFTW," << std::endl
	       << "run configure with --fftw." << std::endl;

#endif // POOMA_FFTW
 
  tester.out() << "-------------------------------------" << std::endl;
  int ret = tester.results("fftwTest1");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: fftwTest1.cpp,v $   $Author: richard $
// $Revision: 1.6 $   $Date: 2004/11/01 18:17:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
