// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#include <fstream>
#include <iostream>
#include <iomanip>   // hex, setw, setfill
#include <string>
#include <algorithm> // replace

#if !POOMA_NO_IOS_HEADER
#  include <ios>
#endif

using std::cout;
using std::endl;
using std::ifstream;
using std::ofstream;
using std::ostream;
using std::hex;
using std::setw;
using std::setfill;
using std::string;

void usage(char *progname)
{
  cout << "Usage: " << progname << "inputfile [outputfile]" << endl;
}

string arrayname(char *filename)
{
  // Find the root name of the file - i.e. peel off the path and any
  // extension. 

  string str(filename);

  // If there are backslashes (as there might be on Windows), replace
  // them by forward slashes.

  std::replace(str.begin(), str.end(), '\\', '/');

  // Find the last slash

  string::size_type begIdx = str.find_last_of("/");

  // If no "/" was found, then the beginning of the word is the
  // beginning of the string, otherwise it is one past the slash.

  if (begIdx == string::npos) 
    begIdx = 0;
  else
    ++begIdx;

  // Find the last '.' in the string

  string::size_type endIdx = str.find_last_of(".");
  
  // If none were found, set endIdx to the string length, otherwise
  // it is set where we want it - one past the last character of the
  // base name.

  string suffix;
  if (endIdx == string::npos) 
    {
      endIdx = str.length();
    }
  else
    {
      string::size_type idx = endIdx + 1;
      suffix = str.substr(idx, str.length() - idx);
    }

  // Return the substring containing the root name:

  return str.substr(begIdx, endIdx - begIdx) + "_" + suffix + "_dump";
}

void dump(char *filename, std::istream &in, std::ostream &out)
{
  int j = 0;
  unsigned char c;

  // Print out the header info

  out << "// Character dump of file " << filename << "\n\n";
  out << "char " << arrayname(filename) << "[] = {\n";

  // Prime the loop.

  in.read((char*)&c,1);
  if (in.gcount() != 0) 
    {
      // Only enter the loop if there are characters to dump.

      out << "  "; // Indent the first line.

      while (true)
        {
          // Write the character to the output stream.

          out << "0x" << setw(2) << setfill('0') << hex << (unsigned int)c;

          // Read another character until we run out of characters.

          in.read((char*)&c,1);
          if (in.gcount() == 0) break;

          // If there is another character, add on the additional
          // formatting.

          out << ",";
          ++j;
          if (j % 4 == 0) out << " ";
          if (j % 8 == 0) out << "\n  ";
        }
    }
  else
    {
      out << "/* Data file was empty */\n";
    }

  // Close out the array definition.

  out << "\n};\n" << endl;
}

int main(int argc, char *argv[])
{
  if (argc == 1 || argc > 3) 
    {
      usage(argv[0]);
      return 1;
    }

  char *infile = argv[1];
  char *outfile = (argc == 3 ? argv[2] : 0);

  ifstream fin(infile, std::ios::binary);

  if (!fin)
    {
      cout << "File: " << infile << " could not be opened.";
      usage(argv[0]);
      return 1;
    }

  if (outfile) 
    {
      ofstream fout(outfile);
      dump(infile, fin, fout);
    }
  else
    {
      dump(infile, fin, cout);
    }
}
