// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// DynamicLayout Dynamic Ops test code
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Domain/Loc.h"
#include "Domain/Interval.h"
#include "Layout/DynamicLayout.h"
#include "Partition/GridPartition.h"
#include "Partition/UniformMapper.h"
#include "Engine/DynamicEngine.h"
#include "Engine/RemoteDynamicEngine.h"
#include "Engine/MultiPatchEngine.h"
#include "Array/Array.h"

#include <iostream>
#include <vector>
using namespace std;

#ifdef POOMA_MESSAGING
typedef MultiPatch<DynamicTag,Remote<Dynamic> > DynamicMultiPatch_t;
#else
typedef MultiPatch<DynamicTag,Dynamic> DynamicMultiPatch_t;
#endif

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);

  // Create the total domain.

  int numContexts = Pooma::contexts();
  Interval<1> x(12*numContexts);
  Interval<1> domain(x);
  
  // Create the block sizes.
  
  Loc<1> blocks(3*numContexts), blocks2(4*numContexts);

  // Create the partitioners.
  
  GridPartition<1> partition(blocks), partition2(blocks2);

  // Create the layouts.
  
  DynamicLayout layout(domain, partition, UniformMapper(partition));
  DynamicLayout layout2(domain, partition2, UniformMapper(partition2));

  tester.out() << "layout1 = " << layout << endl;
  tester.out() << "layout2 = " << layout2 << endl;

  Array<1, double, DynamicMultiPatch_t> a(layout);
  Array<1, double, DynamicMultiPatch_t> aa(layout);
  Array<1, double, DynamicMultiPatch_t> a2(layout2);
  
  Array<1, double, DynamicMultiPatch_t> g(layout);
  Array<1, double, DynamicMultiPatch_t> gg(layout);
  Array<1, double, DynamicMultiPatch_t> g2(layout2);  

  typedef DynamicEvents::PatchID_t             PatchID_t;
  typedef DynamicEvents::CreateSize_t          CreateSize_t;



  // Store some stuff.
  
  typedef Interval<1>::iterator Iterator;
  Iterator b0 = domain[0].begin();
  Iterator e0 = domain[0].end();
  for (Iterator i0=b0; i0!=e0; ++i0)
  {
    int j=i0->first();
     a(j) = aa(j)  = g(j) = gg(j)  = double(j);
     a2(j)=g2(j)=double(j);
  }

  tester.out() << " On layout1, Array a = " << a << endl;

  tester.out() << " destroy element 1 on patch 0 " << endl;

  PatchID_t zero = 0;
  PatchID_t one = 1;
  layout.destroy(Interval<1>(1,1),zero,BackFill());
  layout.sync();

  tester.out() << " after destroy, this is array a " <<endl;

  tester.out()  << a << endl;
  
  tester.out() << " this is array aa, should be the same as a " <<endl;

  tester.out() << aa << endl;
  
  tester.out() << " copy elements 0 and 1 from patch 1 to patch 0. " << endl;

  layout.copy(Interval<1>(0,1),one,zero);

  tester.out() << " after copy, before sync"<<endl;
  
  layout.sync();

  tester.out() << " after sync, this is array a " <<endl;

  tester.out()  << a << endl;
  
  tester.out() << " this is array aa, should be the same as a " <<endl;

  tester.out() << aa << endl;
  
  tester.check(sum((aa - a)*(aa - a)) == 0);

  tester.out() << " create 5 in patch 2 " <<endl;

  layout.create(5,2);
  layout.sync();
    
  for (int i=9;i<=layout.domain().last();++i)
  {
    a(i) = double(i);
    aa(i) = 100 * a(i);
  }

  tester.out() << " this is array a " <<endl;

  tester.out()  << a << endl;

  
  tester.out() << " this is array aa, should be the same as a, "
               << "but with values in patch 2 multiplied by 100." <<endl;


  tester.out() << aa << endl;

#if 0
  // The following are dynamic operations with global domains. They do
  // not work cross context, and it is not clear that we even want
  // this capability, particularly the copy, which mixes global
  // domains and local PatchIDs in a really bad manner.

  tester.out() << " do cross-patch destroy, deleting from 3 to last-2 " <<endl;

  Interval<1> crosspatch(3,layout.domain().last()-2);
  layout.destroy(crosspatch,-1,BackFill());
  layout.sync();
  
  tester.out() << " this is array a " <<endl;

  tester.out()  << a << endl;

  tester.out() << " Array a2, using layout2 " <<a2<<endl;

  tester.out() << "Copy even elements to end " <<endl;

  Range<1> cpcopy(0,11,2);

  layout2.copy(cpcopy,-1);
  
  layout2.sync();

  tester.out() << " Array a2 " <<a2<<endl;

  Array<1, double, DynamicMultiPatch_t> copyg2(g2);
  
  tester.out() << " this is a copy of g2, called copyg2"<<endl;
  
  tester.out() << copyg2<<endl;
  
  layout2.destroy(crosspatch,-1,ShiftUp());
  
  layout2.sync();
  
  tester.out() << " after kill, shiftup on "<<crosspatch<<endl;
  
  tester.out() << " copyg2 "<<copyg2<<endl;
#endif

  layout2.create(6);
  layout2.sync();

  tester.out() << " a.domain() = " <<  a.domain() << std::endl;
  tester.out() << "aa.domain() = " << aa.domain() << std::endl;
  tester.out() << "a2.domain() = " << a2.domain() << std::endl;

  a2 = aa + a * 0.1;

  int ret = tester.results("dynamiclayout_test1");
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: dynamiclayout_test1.cpp,v $   $Author: richard $
// $Revision: 1.7 $   $Date: 2004/11/01 18:16:38 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
