// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Tiny example
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Arrays.h"
#include <iostream>
#include <cmath>

int main(int argc, char* argv[])
{
  // Initialize Pooma.
  Pooma::initialize(argc, argv);

  // Make an array of 100 3D ray vectors.
  Loc<1> patchSize(25);
  UniformGridLayout<1> layout(Interval<1>(100), patchSize, ReplicatedTag());
  Array<1, Vector<3>, MultiPatch<UniformTag,Brick> > rays(layout);
  
  // Set the third component of all of the vectors to zero.
  rays.comp(2) = 0.0;
  
  // Starting some scalar code, must block.
  Pooma::blockAndEvaluate();
  
  // Fill the vectors with a random value for the first component.
  for (int i = 0; i < 100; i++)
    rays(i)(0) = rand() / static_cast<double>(RAND_MAX);
  
  // Define a unit vector pointing in the y direction.
  Vector<3> n(0.0, 1.0, 0.0);
    
  // Set the second component so that the length is one.
  rays.comp(1) = sqrt(1.0 - rays.comp(0) * rays.comp(0));

  // Reflect the rays off of a plane perpendicular to the y axis.  
  rays += -2.0 * dot(rays, n) * n;
  
  // Define a diagonal tensor:
  Tensor<3,double,Diagonal> xyflip2(0.0);
  xyflip2(0,0) = -2.0; xyflip2(1,1) = -2.0;

  // Tensor-Vector dot product multiplies x and y components by -2.0:
  rays = dot(xyflip2, rays);

  // Output the rays.
  std::cout << rays << std::endl;
  
  // Clean up and leave.
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Tiny.cpp,v $   $Author: richi $
// $Revision: 1.12 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
