// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
//   Game of Life Example
//-----------------------------------------------------------------------------

#include <iostream>
#include "Pooma/Arrays.h"
#include "LifeStencil.h"

static const int DefaultWorldSize = 100;
static const int DefaultNumIter   =  10;

//-----------------------------------------------------------------------------
// randomize() : fill array with random values
//-----------------------------------------------------------------------------

void randomize(Array<2> & world, const int worldSize, const int randSeed)
{
  srand(randSeed);
  for (int i=0; i<worldSize; ++i) {
    for (int j=0; j<worldSize; ++j) {
      world(i, j) = (rand() & 0x0080) ? 1 : 0;
    }
  }
}

//-----------------------------------------------------------------------------
// Main program
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

  // get problem size, number of iterations, and RNG seed
  if (argc != 4){
    std::cerr << "usage: Life worldSize numIter randSeed" << std::endl;
    exit(1);
  }
  int worldSize = atoi(argv[1]);
  int numIter   = atoi(argv[2]);
  int randSeed  = atoi(argv[3]);

  // create and size the world array, and create a stencil
  Array<2> world(worldSize, worldSize);
  Stencil<LifeStencil> stencil;

  // describe the interior of the world
  Interval<1> interior_1(1, worldSize-2);
  Interval<2> interior_2(interior_1, interior_1);

  // set array element values
  randomize(world, worldSize, randSeed);

  // update elements
  std::cout << " USING THE INTERVAL ON THE LEFT " << std::endl;
  int i;
  for (i=0; i<numIter; ++i) {
    world(interior_2) = stencil(world);
    Pooma::blockAndEvaluate();
    std::cout << "== " << i    << " ==" << std::endl
	      << world << std::endl << std::endl;
  }

  // re-set world elements
  randomize(world, worldSize, randSeed);

  // update interior elements
  std::cout << " USING THE INTERVAL ON THE RIGHT " << std::endl;
  for (i=0; i<numIter; ++i) {
    world(interior_2) = stencil(world, interior_2);
    Pooma::blockAndEvaluate();
    std::cout << "== " << i    << " ==" << std::endl
	      << world << std::endl << std::endl;
  }
  
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Life.cpp,v $
// $Revision: 1.8 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
