// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// RBJacobi2.cpp is used in Tutorial 2 to show how multi-dimensional intervals
// can be used as subscripts.
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"

#include <iostream>

// The size of each side of the domain. Must be even.
const int N = 20;

// Apply a Jacobi iteration on the given domain.
void
ApplyJacobi(
    const Array<2>       &    V,              // to be relaxed
    const Array<2>       &    b,              // fixed term
    const Range<2>       &    IJ              // region of calculation
){
    V(IJ) = 0.25 * (V(IJ+Loc<2>(1,0)) + V(IJ+Loc<2>(-1,0)) +
                    V(IJ+Loc<2>(0,1)) + V(IJ+Loc<2>(0,-1)) - b.read(IJ));
}

int
main(
    int                 argc,           // argument count
    char *              argv[]          // argument vector
){
    // Initialize Pooma.
    Pooma::initialize(argc,argv);

    // The calculation domain.
    Interval<2> calc( Interval<1>(1, N-2), Interval<1>(1, N-2) );

    // The domain with guard cells on the boundary.
    Interval<2> guarded( Interval<1>(0, N-1) , Interval<1>(0, N-1) );

    // The array we'll be solving for.
    Array<2> V(guarded);
    V = 0.0;

    // The right hand side of the equation.
    Array<2> b(calc);
    b = 0.0;

    // Must block since we're doing some scalar code (see Tutorial 4).
    Pooma::blockAndEvaluate();
    b(N/2, N/2) = -1.0;

    // The interior domain, now with stride 2.
    Range<2> IJ( Range<1>(1, N-3, 2), Range<1>(1, N-3, 2) );

    // Iterate 100 times.
    for (int i=0; i<100; ++i)
    {
        ApplyJacobi(V, b, IJ);
        ApplyJacobi(V, b, IJ+Loc<2>(1, 1)); 
        ApplyJacobi(V, b, IJ+Loc<2>(1, 0));
        ApplyJacobi(V, b, IJ+Loc<2>(0, 1));
    }

    // Print out the result.
    std::cout << V << std::endl;

    // Clean up and report success
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RBJacobi2.cpp,v $   $Author: richard $
// $Revision: 1.8 $   $Date: 2004/11/01 18:16:00 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
