// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// RBJacobi.cpp is used in Tutorial 2 to show how arrays are passed as
// parameters, what the difference between Array and ConstArray is, and how
// to subscript arrays with ranges to select non-contiguous elements.  See
// 'RBJacobi2.cpp' in examples/RBJacobi2 for a fully 2D version of this code.
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"

#include <iostream>

// The size of each side of the domain.
const int N = 20;

// Apply a Jacobi iteration on the given domain.
void
ApplyJacobi(
    const Array<2>       &    V,              // to be relaxed
    const Array<2>       &    b,              // fixed term
    const Range<1>       &    I,              // range on first axis
    const Range<1>       &    J               // range on second axis
){
    V(I,J) = 0.25 * (V(I+1,J) + V(I-1,J) + V(I,J+1) + V(I,J-1) - b.read(I,J));
}

int
main(
    int                 argc,           // argument count
    char *              argv[]          // argument list
){
    // Initialize Pooma.
    Pooma::initialize(argc, argv);

    // The array we'll be solving for.
    Array<2> V(N, N);
    V = 0.0;

    // The right hand side of the equation.
    Array<2> b(N,N);
    b = 0.0;

    // Must block since we're doing some scalar code (see Tutorial 4).
    Pooma::blockAndEvaluate();
    b(N/2, N/2) = -1.0;

    // The interior domain, now with stride 2.
    Range<1> I(1, N-3, 2), J(1, N-3, 2);

    // Iterate 100 times.
    for (int iteration=0; iteration<100; ++iteration)
    {
        // red
        ApplyJacobi(V, b, I,   J);
        ApplyJacobi(V, b, I+1, J+1);
        // black
        ApplyJacobi(V, b, I+1, J);
        ApplyJacobi(V, b, I,   J+1);
    }

    // Print out the result.
    std::cout << V << std::endl;

    // Clean up and report success.
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RBJacobi.cpp,v $   $Author: richard $
// $Revision: 1.15 $   $Date: 2004/11/01 18:15:58 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
