// -*- C++ -*-
//
// Copyright (C) 2004  Richard Guenther
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// Example on how to use PETSc linear solvers from inside a Pooma program.
// This solves the Poisson equation for a density distribution in a 3D
// Array to create the corresponding gravitational potential.

#include <iostream>
#include "Transform/PETSc.h"

#include "petscda.h"
#include "petscksp.h"

void doit()
{
  typedef Array<2, double, MultiPatch<GridTag, Remote<Brick> > > Array_t;
  typedef Array_t::Layout_t::const_iterator layout_iter;

  Interval<2> domain(8, 8);
  Loc<2> blocks = makeRBlocks(domain, Pooma::contexts());
  GridLayout<2> layout(domain, blocks, GuardLayers<2>(1), DistributedTag());

  // Create DA
  Pooma::PoomaDA<2> da(layout, DA_NONPERIODIC, DA_STENCIL_STAR, 1);

  // Assemble matrix
  Mat jac;
  DAGetMatrix(da, MATMPIAIJ, &jac);
  {
    int mx, my;
    PetscScalar Hx, Hy, HxdHy, HydHx;
    DAGetInfo(da,0,&mx,&my,PETSC_NULL,0,0,0,0,0,0,0);
    Hx = 1.0 / (PetscReal)(mx-1); Hy = 1.0 / (PetscReal)(my-1);
    HxdHy = Hx/Hy; HydHx = Hy/Hx;
    int xs, ys, xm, ym;
    MatStencil row,col[5];
    DAGetCorners(da,&xs,&ys,PETSC_NULL,&xm,&ym,PETSC_NULL);
    PetscScalar v[5];
    for (int j=ys; j<ys+ym; j++){
      for(int i=xs; i<xs+xm; i++){
	row.i = i; row.j = j;
	if (i==0 || j==0 || i==mx-1 || j==my-1) {
	  v[0] = 2.0*(HxdHy + HydHx);
	  MatSetValuesStencil(jac,1,&row,1,&row,v,INSERT_VALUES);
	} else {
	  v[0] = -HxdHy;              col[0].i = i;   col[0].j = j-1;
	  v[1] = -HydHx;              col[1].i = i-1; col[1].j = j;
	  v[2] = 2.0*(HxdHy + HydHx); col[2].i = i;   col[2].j = j;
	  v[3] = -HydHx;              col[3].i = i+1; col[3].j = j;
	  v[4] = -HxdHy;              col[4].i = i;   col[4].j = j+1;
	  MatSetValuesStencil(jac,1,&row,5,col,v,INSERT_VALUES);
	}
      }
    }
  }
  MatAssemblyBegin(jac,MAT_FINAL_ASSEMBLY);
  MatAssemblyEnd(jac,MAT_FINAL_ASSEMBLY);

  // problem
  Array_t rh(layout), ph(layout);
  rh(rh.totalDomain()) = 1.0;
  ph(ph.totalDomain()) = 0.0;
  Pooma::pinfo << "Density distribution:\n" << rh << std::endl;

  // assemble rhs and solution
  Vec pph, prh;
  DACreateGlobalVector(da, &prh);
  VecDuplicate(prh, &pph);

  da.assign(prh, rh);

  // create solver
  KSP ksp;
  KSPCreate(PETSC_COMM_WORLD, &ksp);
  KSPSetTolerances(ksp, 1e-10, 1e-5, 1e5, 10);
  KSPSetOperators(ksp, jac, jac, DIFFERENT_NONZERO_PATTERN);
  KSPSetFromOptions(ksp);

  // solve and copy solution
  KSPSetRhs(ksp, prh);
  KSPSetSolution(ksp, pph);
  KSPSolve(ksp);
  da.assign(ph, pph);

  // show soultion
  Pooma::pinfo << "Solution:\n" << ph << std::endl;
}

int main(int argc, char **argv)
{
  Pooma::initialize(argc, argv);

  PetscSetCommWorld(MPI_COMM_WORLD);
  PetscInitialize(&argc, &argv, NULL, NULL);

  doit();

  // cleanup
  PetscFinalize();
  Pooma::finalize();
  return 0;
}

