// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// NinePointMatrix
//-----------------------------------------------------------------------------

#ifndef POOMA_NINEPOINTMATRIX_H
#define POOMA_NINEPOINTMATRIX_H

#include "NinePoint.h"

struct MatVecNinePoint
{
  template<class Tag>
  static void
  apply(
	const Array<2, double, Tag> &y,
	const Array<2, double, Tag> &x,
	const Array<2, NinePoint, Tag> &values
	)
  {

    Interval<2> domain = values.domain();

    int i0, i1;
    int b0 = domain[0].first();
    int b1 = domain[1].first();
    int e0 = domain[0].last();
    int e1 = domain[1].last();

    for (i0 = b0 + 1; i0 <= e0 - 1; ++i0)
    {
      for (i1 = b1 + 1; i1 <= e1 - 1; ++i1)
      {
	y(i0, i1) =
	  values(i0, i1).southwest() * x(i0 - 1, i1 - 1) +
	  values(i0, i1).south() * x(i0, i1 - 1) +
	  values(i0, i1).southeast() * x(i0 + 1, i1 - 1) +
	  values(i0, i1).west() * x(i0 - 1, i1) +
	  values(i0, i1).center() * x(i0, i1) +
	  values(i0, i1).east() * x(i0 + 1, i1) +
	  values(i0, i1).northwest() * x(i0 - 1, i1 + 1) +
	  values(i0, i1).north() * x(i0, i1 + 1) +
	  values(i0, i1).northeast() * x(i0 + 1, i1 + 1);
      }
    }
  } // apply()

  inline int lowerExtent(int) const { return 1; }
  inline int upperExtent(int) const { return 1; }

};

template<class EngineTag>
class NinePointMatrix
{
public:
  typedef Array<2, NinePoint, EngineTag> Values_t;

  inline Values_t &values() { return values_m; }
  inline const Values_t &values() const { return values_m; }

  template<class Layout, class Domain>
  NinePointMatrix(const Layout &layout, const Domain &domain)
    : values_m(layout), domain_m(domain)
  {
  }

  static bool needGuards() { return true; }
  static GuardLayers<2> guardLayers() { return GuardLayers<2>(1); }

  void operator()(
		  const Array<2, double, EngineTag> &x,
		  const Array<2, double, EngineTag> &y
		  ) const
  {
    GuardedPatchEvaluator<MainEvaluatorTag>::evaluate(y, x, values_m,
						      MatVecNinePoint(),
						      domain_m);
  }

private:
  Values_t values_m;
  Interval<2> domain_m;
  double zero_m;
};

#endif // POOMA_NINEPOINTMATRIX_H
