// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// FivePointIC0Stencil
//-----------------------------------------------------------------------------

#ifndef POOMA_FIVEPOINTIC0STENCIL_H
#define POOMA_FIVEPOINTIC0STENCIL_H

#include <iostream>

class FivePointIC0Stencil
{

public:

  FivePointIC0Stencil () {}
  FivePointIC0Stencil (double s, double w, double c)
  {
    south_m = s;
    west_m = w;
    center_m = c;
  }

  FivePointIC0Stencil &operator= (const FivePointIC0Stencil &in)
  {
    south_m = in.south_m;
    west_m = in.west_m;
    center_m = in.center_m;
    return *this;
  }

  // jacobi puts the reciprocal of the center element in the center
  // of the new stencil, then divides through by the negative of
  // the center -- the Jacobi iteration is x = D^{-1}Bx + D^{-1}b,
  // where A = D - B; use the center coefficient for the rhs calc,
  // the other two for the x multiply
  
  FivePointIC0Stencil jacobi () 
  {
    return FivePointIC0Stencil (- south_m / center_m,
       - west_m / center_m, 1.0 / center_m);
  }

  FivePointIC0Stencil kludge () 
  {
    return FivePointIC0Stencil (south_m, west_m, 1.0 / center_m);
  }

  inline double &south() { return south_m; }
  inline double &west() { return west_m; }
  inline double &center() { return center_m; }

  inline const double &south() const { return south_m; }
  inline const double &west() const { return west_m; }
  inline const double &center() const { return center_m; }

  // number of nonzero entries; assume that center is nonzero
  int nnz() 
  {
    int res = 1;
    if (south_m != 0) res = res + 1;
    if (west_m != 0) res = res + 1;
    return res;
  }

   void print () 
  {
    std::cout << "( " << south_m << ", " << west_m << ", " << center_m <<
      ") " << std::endl;
    return;
  }
   ~FivePointIC0Stencil() {};

private:

  double south_m;
  double west_m;
  double center_m;

}; // end class FivePointIC0Stencil

std::ostream& operator<< (std::ostream& s, const FivePointIC0Stencil& c)
{
  return s << "( " << c.south() << ", " << c.west() << ", " <<
     c.center() << ") " ;
}

#endif // POOMA_FIVEPOINTIC0STENCIL_H
