// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Contents:
//   This example illustrates how to use the IndexFunctionEngine to fill in
// arrays with index values in a data-parallel fashion.
//
// In POOMA r1, it was typical to use initialization statements like:
// a(I,J) = I+sin(J);
// in order to initialize an array with a function of its indices.
//
// POOMA r2 doesn't yet implement index objects, but supports a flexible engine
// types that allow us to express arbitrarily complicated computed values.
// The following example creates a function that returns an array that computes
// vectors, allowing the user to write:
//
// a = iota(domain).comp(0) + sin(iota(domain).comp(1));
//
//-----------------------------------------------------------------------------

// include files

// Bring in Pooma Array machinery.
#include "Pooma/Arrays.h"
#include "Pooma/Indices.h"
#include "Pooma/Fields.h"

//-----------------------------------------------------------------------------
// Main program.
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

  Interval<1> I(1,10), J(1,10);
  Interval<2> domain(I,J);

  Loc<2> patches(2,2);
  UniformGridLayout<2> layout(domain, patches, ReplicatedTag());
  Array<2,double,MultiPatch<UniformTag,Brick> > a(layout);

  Iota<2>::Iota_t  ij(iota(domain));
  Iota<2>::Index_t I1(ij.comp(0));
  Iota<2>::Index_t J1(ij.comp(1));

  a = 2.0*I1;
  std::cout << "0 component:" << std::endl
	    << a << std::endl;

  a = 3.0*J1;
  std::cout << "1 component:" << std::endl
	    << a << std::endl;

  a = sin( dot(Vector<2,double>(0.2, 0.3), ij ) );
  std::cout << "expression of components:" << std::endl
	    << a << std::endl;

  // Create the (uniform, logically rectilinear) mesh.
  Vector<2> origin(0.0, 0.0), spacings(0.2, 0.3);
  typedef UniformRectilinearMesh<2> Mesh_t;
  Mesh_t mesh(domain, origin, spacings);

  Centering<2> cell = canonicalCentering<2>(CellType, Continuous);
  Field<Mesh_t, double, MultiPatch<UniformTag,Brick> > u(cell, layout, mesh);

  Interval<2> d2 = u.physicalDomain();

  Iota<2>::Index_t I2(iota(d2).comp(0));
  Iota<2>::Index_t J2(iota(d2).comp(1));

  u = 0.0;
  u = sin( I2*0.2 ) + J2*5;

  std::cout << "expression with a field:" << std::endl
	    << u << std::endl;

}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Coordinate.cpp,v $   $Author: richard $
// $Revision: 1.15 $   $Date: 2004/11/01 18:15:35 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
