// -*- C++ -*-
//
// Copyright (C) 2002  Jeffrey Oldham
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// Chevron Kernel Written Using POOMA's Proposed Field Abstraction

#include <iostream>
#include <cstdlib>
#include "Pooma/Fields.h"
// linear algebra files:
#include "tnt/tnt.h"
#include "tnt/vec.h"
#include "tnt/cmat.h"
#include "tnt/lu.h"

// This program implements "Implementation of a Flux-Continuous Fnite
// Difference Method for Stratigraphic, Hexahedron Grids," by
// S. H. Lee, H. Tchelepi, and L. J. DeChant, \emph{1999 SPE Reservoir
// Simulation Symposium}, SPE (Society of Petroleum Engineers) 51901.

// Preprocessor symbols:
// PSEUDOCODE: Do not define this symbol.  Surrounds desired code to
//	       deal with different granularity fields.
// DEBUG: If defined, print some information about internal program
//	  values.


/** QUESTIONS **/

// o. According to my understanding, the Chevron algorithm should be
//    imbedded inside a loop of some type that repeatedly updates the
//    coordinates.
// o. I omitted a separate coordinates field, presumably updated each
//    iteration, in favor of using the mesh.  Since I do not know how
//    the coordinates are updated, I omitted updating the mesh.
// o. Creating non-canonical edge and face centerings requires
//    dimension-dependent code.  Is this acceptable?


/** UNFINISHED WORK **/

// o meshLayout.unitCoordinateNormals()
// o field.mesh()
// o field.mesh().normals()
// o field.mesh().normals().signedMagnitude()
// o sum(field, vector<FieldOffsetList>, centering)

/** EXPLANATIONS **/

// o Centering<Dim> canonicalCentering<Dim>(CellType, Continuous):
//    returns a centering object for a cell-centered field with one
//    value at the cell's center (in logical coordinate space)
// o subcell: This centering contains four cell-centered values at
//    positions (0.25, 0.25), (0.25, 0.75), (0.75, 0.75), (0.75, 0.25).  
//    Since this centering is not a canonical centering, it must be
//    constructed.  To do so, we start with a cell-centered centering
//    without any values and repeatedly add values.  The orientation,
//    ignored for cell-centered values, indicates which coordinate values
//    are fixed and which are not.  Using a (1,...,1) indicates that
//    all coordinate values may be changed.
// o spoke: This face-centering has two values on each face.  It, too,
//    has to be constructed since it is not a normal centering.
// o The Chevron algorithm first solves a linear program.  I have
//    omitted since computation since it does not illustrate field
//    computations.
// o replicate(field, std::vector<FieldOffsetList>, centering): This
//    function, syntactic sugar for a nearest neighbors computation,
//    copies the field values to the positions indicated by the
//    std::vector<FieldOffsetList>.  Each field value is copied to one or
//    more values.  replicate() could be replaced by sum(), but the
//    latter function has an unnecessary loop since each output value
//    equals one input value.
// o nearestNeighbors(inputCentering, outputCentering): This function
//    returns a std::vector of FieldOffsetList's, one for each output
//    value specified by the given output centering.  For each output
//    value, the closest input values, wrt Manhattan distance, are
//    returned.  Eventually, these may be pre-computed or cached to
//    reduce running time.
// o nearestNeighbors(input centering, FieldOffset, offset's centering
//    [, bool]):  This function returns a FieldOffsetList with entries for
//    the closest input values, wrt Manhattan distance, to the value
//    specified by the FieldOffset and the offset centering.  The
//    optional fourth parameter indicates only values from the FieldOffset's
//    cell should be returned.
// o meshLayout.unitCoordinateNormals(): This returns a discontinuous
//    face-centered field with unit-length normals all pointing in
//    positive directions.
// o field.mesh(): Returns the mesh object associated with the field.
// o spokeFlux.mesh().normals(): Returns a face-centered field of
//    normal vectors perpendicular to each face.  The magnitude of each
//    normal equals the face's area/volume.
// o spokeFlux.mesh().normals().signedMagnitude(): Returns a
//    face-centered field of scalars, each having absolute value
//    equalling the face's area/volume and sign equalling whether the
//    face's normal is in a positive direction, e.g., the positive
//    x-direction vs. the negative x-direction.
// o sum(field, vector<FieldOffsetList>, centering): this
//    parallel-data statement adds the values indicated in the
//    FieldOffsetList to form each output value


/** DESIGN DECISIONS **/

// o. Several different fields can share the same mesh.
// o. Meshes can be queried without a field.

/** THE PROGRAM **/


// Use a linear algebra computation that stores the pressure gradient
// values in the four quadrants around the current vertex (loc).

template <int Dim>
struct ComputeGradientsInfo
{
  void scalarCodeInfo(ScalarCodeInfo &info) const
  {
    // ComputeGradients::operator() has 4 arguments: vertices to
    // iterate over and three fields.

    info.arguments(5);

    // Only the first such argument is changed.

    info.write(0, false);
    info.write(1, true);
    info.write(2, false);
    info.write(3, false);
    info.write(4, false);

    // Does the operation index neighboring cells, i.e., do we need to
    // update the internal guard layers?
   
    info.useGuards(0, false);
    info.useGuards(1, true);
    info.useGuards(2, true);
    info.useGuards(3, true);
    info.useGuards(4, true);
    
    info.dimensions(Dim);

    for (int i = 0; i < Dim; ++i) {
      info.lowerExtent(i) = 1;	// We access neighbors.
      info.upperExtent(i) = 0;
    }
  }
};


template <int Dim>
struct ComputeGradients
  : public ComputeGradientsInfo<Dim>
{

  ComputeGradients(const Centering<Dim> &disspoke,
		   const FieldOffsetList<Dim> &gradients,
		   const int nuFluxPoints,
		   const std::vector<FieldOffsetList<Dim> > &disFluxPoints,
		   const Centering<Dim> &subcell)
    : disspoke_m(disspoke), gradients_m(gradients),
      nuFluxPoints_m(nuFluxPoints), disFluxPoints_m(disFluxPoints),
      subcell_m(subcell)
  {
    PAssert(gradients_m.size() * Dim == nuFluxPoints_m * 2);
  }

  // FIXME: Perhaps we want to modify ScalarCode to take a first
  // FIXME: argument of a centering.  In the meantime, we just use a
  // FIXME: fake field.

  template <class F1, class F2, class F3, class F4, class F5>
  inline
  void operator()(const F1 &vertexField,
		  F2 &pressureGradient,
		  const F3 &faceDistance,
		  const F4 &directedPermeability,
		  const F5 &pressure,
		  const Loc<Dim> &loc) const
  {
    const int nuRows = (1 << Dim) * Dim;
    TNT::Matrix<double> A(nuRows, nuRows, 0.0);
    TNT::Vector<double> rhs(nuRows, 0.0);
    TNT::Vector<TNT::Subscript> ipiv;	// ignored

    // Assign values to the matrix A and vector rhs.

    for (int faceIndex = nuFluxPoints_m-1; faceIndex >= 0; --faceIndex) {
      // Work on the "positive" side of the face.
      FieldOffset<Dim> fo = disFluxPoints_m[faceIndex][0];
      int columnNu =
	findIndex(gradients_m,
		  nearestNeighbors(pressureGradient.centering(),
				   fo, disspoke_m, true)[0]);
      // The column number is the pressure gradient corresponding to the
      // "positive" side of the face.

      for (int i = 0; i < Dim; ++i)
	A[faceIndex][columnNu] =
	  directedPermeability(nearestNeighbors(directedPermeability.centering(),
						fo, disspoke_m, true)[0],
			       loc+fo.cellOffset())(i);
      A[faceIndex+nuFluxPoints_m][columnNu] =
	faceDistance(nearestNeighbors(faceDistance.centering(),
				      fo, disspoke_m, true)[0],
		     loc+fo.cellOffset());
      rhs[faceIndex+nuFluxPoints_m] -=
	pressure(nearestNeighbors(pressure.centering(),
				  fo, disspoke_m, true)[0],
		 loc+fo.cellOffset());

      // Work on the "negative" side of the face.
      fo = disFluxPoints_m[faceIndex][1];
      columnNu =
	findIndex(gradients_m,
		  nearestNeighbors(pressureGradient.centering(),
				   fo, disspoke_m, true)[0]);
      // The column number is the pressure gradient corresponding to the
      // "positive" side of the face.

      for (int i = 0; i < Dim; ++i)
	A[faceIndex][columnNu] =
	  -directedPermeability(nearestNeighbors(directedPermeability.centering(),
						 fo, disspoke_m, true)[0],
				loc+fo.cellOffset())(i);
      A[faceIndex+nuFluxPoints_m][columnNu] =
	-faceDistance(nearestNeighbors(faceDistance.centering(),
				       fo, disspoke_m, true)[0],
		      loc+fo.cellOffset());
      rhs[faceIndex+nuFluxPoints_m] -=
	-pressure(nearestNeighbors(pressure.centering(),
				   fo, disspoke_m, true)[0],
		  loc+fo.cellOffset());
    }

    // Solve for the pressure gradients.

    TNT::LU_solve(A, ipiv, rhs);

    // Now, rhs has the pressure gradients.

    for (int faceIndex = nuFluxPoints_m-1; faceIndex >= 0; --faceIndex)
      for (int i = 0; i < Dim; ++i)
	pressureGradient(gradients_m[faceIndex], loc)(i) = rhs[faceIndex+i];

    return;
  }

  // Return the index of the specified field offset in the given list.
  // Return a negative number if not found.

  static
  inline int
  findIndex(const FieldOffsetList<Dim> &vec,
	    const FieldOffset<Dim> &fo)
  {
    int indx;
    for (indx = vec.size()-1;
	 indx >= 0 && vec[indx] != fo;
	 --indx)
      ;
    return indx;
  }

private:

  // Discontinuous spokes.
  const Centering<Dim> &disspoke_m;

  // The pressure gradients.
  const FieldOffsetList<Dim> &gradients_m;

  // The number of flux points for a cell.
  const int nuFluxPoints_m;

  // For every i, disFluxPoints_m[i] is two discontinuous positions on
  // "either side" of the face represented by the flux points
  // fluxPoints[i].
  const std::vector<FieldOffsetList<Dim> > &disFluxPoints_m;

  // One cell value in each quadrant.
  const Centering<Dim> &subcell_m;
};


int main(int argc, char *argv[])
{
  // Set up the Pooma library.
  Pooma::initialize(argc,argv);
#ifdef DEBUG
  std::cout << "Start program." << std::endl;
#endif // DEBUG


  /* DECLARATIONS */

  // Create a simple layout.
  const unsigned Dim = 2;		// Work in a 2D world.
  const unsigned nXs = 5;		// number of horizontal vertices
  const unsigned nYs = 4;		// number of vertical vertices
  Interval<Dim> meshDomain;
  meshDomain[0] = Interval<1>(nXs);
  meshDomain[1] = Interval<1>(nYs);
  DomainLayout<Dim> meshLayout(meshDomain, GuardLayers<Dim>(1));

  // Preparation for Field creation.

  Vector<Dim> origin(0.0);
  Vector<Dim> spacings(1.0,1.0);
  typedef UniformRectilinearMesh<Dim, double> Geometry_t;
  typedef Field<Geometry_t, double, Brick> Fields_t;
  typedef Tensor<Dim,double,Full> Tensor_t;
  typedef Field<Geometry_t, Tensor_t, Brick> FieldT_t;
  typedef Field<Geometry_t, Vector<Dim>, Brick> Fieldv_t;

  // Cell-centered Fields.

  Centering<Dim> cell = canonicalCentering<Dim>(CellType, Continuous);
  FieldT_t permeability		(cell, meshLayout, origin, spacings);
  Fields_t pressure		(cell, meshLayout, origin, spacings);
  Fields_t totalFlux		(cell, meshLayout, origin, spacings);

  // Subcell-centered Field.

  typedef Centering<Dim>::Orientation Orientation;
  typedef Centering<Dim>::Position Position;
  Position position;
  Centering<Dim> subcell(CellType, Continuous);
  position(0) = 0.25;
  position(1) = 0.25; subcell.addValue(Orientation(1), position);
  position(1) = 0.75; subcell.addValue(Orientation(1), position);
  position(0) = 0.75; subcell.addValue(Orientation(1), position);
  position(1) = 0.25; subcell.addValue(Orientation(1), position);
  Fieldv_t pressureGradient	(subcell, meshLayout, origin, spacings);

  // Spoke-centered Field.

  Centering<Dim> spoke(FaceType, Continuous);
  Orientation orientation;
  // NOTE: This code is not dimension-independent.
  for (int zeroFace = 0; zeroFace < 2; ++zeroFace) {
    orientation = 1; orientation[zeroFace] = 0;
    position(zeroFace) = 0.0;
    position(1-zeroFace) = 0.25; spoke.addValue(orientation, position);
    position(1-zeroFace) = 0.75; spoke.addValue(orientation, position);
  }
  Fields_t spokeFlux		(spoke, meshLayout, origin, spacings);

  Centering<Dim> disspoke(FaceType, Discontinuous);
  // NOTE: This code is not dimension-independent.
  for (int zeroFace = 0; zeroFace < 2; ++zeroFace) {
    orientation = 1; orientation[zeroFace] = 0;
    position(zeroFace) = 0.0;
    position(1-zeroFace) = 0.25; disspoke.addValue(orientation, position);
    position(1-zeroFace) = 0.75; disspoke.addValue(orientation, position);
    position(zeroFace) = 1.0;
    position(1-zeroFace) = 0.25; disspoke.addValue(orientation, position);
    position(1-zeroFace) = 0.75; disspoke.addValue(orientation, position);
  }

  // Face-centered.

  Centering<Dim> disFace = canonicalCentering<Dim>(FaceType, Discontinuous);
  Fieldv_t directedPermeability	(disFace, meshLayout, origin, spacings);
  // \gamma_{i,j} = K_i^t \dot \hat{n}_j
  Fields_t faceDistance		(disFace, meshLayout, origin, spacings);
  // distance from cell center to face center


  /* INITIALIZATION */

#ifdef PSEUDOCODE
  // Initialize tensors.
  // Initialize the pressures.
  // Initialize coordinates.
#endif // PSEUDOCODE


  /* COMPUTATION */

  // Compute pressureGradients by simultaneously solving several
  // linear equations.  The operands have different centerings.

  // \Gamma is used in the flux continuity equations.
  directedPermeability =
    dot(replicate(permeability,
		  nearestNeighbors(permeability.centering(),
				   directedPermeability.centering()),
		  directedPermeability.centering()),
	meshLayout.unitCoordinateNormals());

#ifdef PSEUDOCODE
  // These distances are used in the pressure continuity equations.
  faceDistance = face-centered-positions -
    replicate(cell-centered-positions,
	      nearestNeighbors(cell, faceDistance.centering()),
	      faceDistance.centering());
#endif // PSEUDOCODE
  
  const Centering<Dim> vert = canonicalCentering<Dim>(VertexType, Continuous);
  PInsist(vert.size() == 1, "The vertex centering has too many values.");

  const FieldOffsetList<Dim> gradients =
    nearestNeighbors(pressureGradient.centering(),
		     FieldOffset<Dim>(Loc<Dim>(0)) /* cell origin */, vert);
  // gradients's order of pressure gradients will be used for the
  // matrix and rhs.

  const FieldOffsetList<Dim> fluxPoints =
    nearestNeighbors(spokeFlux.centering(),
		     FieldOffset<Dim>(Loc<Dim>(0)) /* cell origin */, vert);
  // fluxPoints has locations for all faces incident to the vertex.

  const int nuFluxPoints = fluxPoints.size();

  const std::vector<FieldOffsetList<Dim> > disFluxPoints =
    nearestNeighbors(disspoke, fluxPoints, spokeFlux.centering());
  // For every i, disFluxPoints[i] is two discontinuous positions on
  // "either side" of the face represented by fluxPoints[i].

  typedef ComputeGradients<Dim> CG_t;
  CG_t cG(disspoke, gradients, nuFluxPoints, disFluxPoints, subcell);
  ScalarCode<CG_t> computeGradients(cG);

  // FIXME: Use an otherwise unused field for the ScalarCode
  // FIXME: iteration over vertices.
  Fields_t vertexField(canonicalCentering<Dim>(VertexType, Continuous),
		       meshLayout, origin, spacings);

  computeGradients(vertexField, pressureGradient,
		   faceDistance, directedPermeability, pressure);

  // Compute the spoke fluxes.

  spokeFlux = 
    dot(replicate(pressureGradient,
		  nearestNeighbors(pressureGradient.centering(),
				   spokeFlux.centering(),
				   true),
		  spokeFlux.centering()),
	replicate(directedPermeability,
		  nearestNeighbors(directedPermeability.centering(),
				   spokeFlux.centering(),
				   true),
		  spokeFlux.centering()));

  // Sum the spoke fluxes into a cell flux.

  // Q = \sum_{faces f of cell} sign_f area_f \sum_{subfaces sf of f} q_{sf}
  // We compute this in three steps:
  // 1. Add together the flux values on each face to form a
  //    face-centered field.
  // 2. Multiply each value by the magnitude of the face's normal.
  // 3. Add together each face's value.

  totalFlux =
    sum(spokeFlux.mesh().normals().signedMagnitude() *
	sum(spokeFlux,
	    nearestNeighbors(spokeFlux.centering(), disFace),
	    disFace),
	nearestNeighbors(disFace, totalFlux.centering()),
	totalFlux.centering());


  /* TERMINATION */

  std::cout << "total flux:\n" << totalFlux << std::endl;
#ifdef DEBUG
  std::cout << "End program." << std::endl;
#endif // DEBUG
  Pooma::finalize();
  return EXIT_SUCCESS;
}
