// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// ----------------------------------------------------------------------------
// Jacobi Laplace solver illustrating the use of a custom field stencil
// and the implementation of custom boundary conditions. This version
// differs from the one in examples/Field/Laplace in that the boundary
// conditions are associated with the operator, not the field.
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Clock.h"

#include <iostream>

// Convenience typedefs.

typedef ConstField<
  DiscreteGeometry<Vert, UniformRectilinearMesh<2> > > ConstFieldType_t;

typedef Field<
  DiscreteGeometry<Vert, UniformRectilinearMesh<2> > > FieldType_t;

// The boundary condition.

class PositionFaceBC : public BCondCategory<PositionFaceBC>
{
public:

  PositionFaceBC(int face) : face_m(face) { }

  int face() const { return face_m; }

private:

  int face_m;
};

template<>
class BCond<FieldType_t, PositionFaceBC> 
  : public FieldBCondBase<FieldType_t>
{
public:

  BCond(const FieldType_t &f, const PositionFaceBC &bc)
  : FieldBCondBase<FieldType_t>
      (f, f.totalDomain()), bc_m(bc) { }

  void applyBoundaryCondition()
  {
    int d = bc_m.face() / 2;
    int hilo = bc_m.face() & 1;
    int layer;
    Interval<2> domain(subject().totalDomain());
    if (hilo)
      layer = domain[d].last();
    else
      layer = domain[d].first();
      
    domain[d] = Interval<1>(layer, layer);
    subject()(domain) = 100.0 * subject().x(domain).comp(0) *
      subject().x(domain).comp(1);
  }
  
  FieldBCondBase<FieldType_t> *retarget(const FieldType_t &f) const
  {
    return new BCond<FieldType_t, PositionFaceBC>(f, bc_m);
  }

private:

  PositionFaceBC bc_m;
};

// The stencil.
  
class Laplacian
{
public:

  typedef Vert OutputCentering_t;
  typedef double OutputElement_t;

  int lowerExtent(int) const { return 1; }
  int upperExtent(int) const { return 1; }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.25 * (f(i1 + 1, i2) + f(i1 - 1, i2) + 
      f(i1, i2 + 1) + f(i1, i2 - 1));
  }
  
  template<class F>
  static void applyBoundaryConditions(const F &f)
  {
    for (int i = 0; i < 4; i++)
      {
        BCondItem *bc = PositionFaceBC(i).create(f);
        bc->applyBoundaryCondition();
        delete bc;
      }
  }
};

void applyLaplacian(const FieldType_t &l, const FieldType_t &f)
{
  Laplacian::applyBoundaryConditions(f);
  l = FieldStencil<Laplacian>()(f);
}

int main(
    int argc,
    char *argv[]
){
    // Set up the library
    Pooma::initialize(argc,argv);

    // Create the physical domains:

    // Set the dimensionality:
    const int nVerts = 100;
    Loc<2> center(nVerts / 2, nVerts / 2);
    Interval<2> vertexDomain(nVerts, nVerts);

    // Create the (uniform, logically rectilinear) mesh.
    Vector<2> origin(1.0 / (nVerts + 1)), spacings(1.0 / (nVerts + 1));
    typedef UniformRectilinearMesh<2> Mesh_t;
    Mesh_t mesh(vertexDomain, origin, spacings);
    
    // Create a geometry object with 1 guard layer to account for
    // stencil width:
    typedef DiscreteGeometry<Vert, UniformRectilinearMesh<2> > Geometry_t;
    Geometry_t geom(mesh, GuardLayers<2>(1));

    // Create the Fields:

    // The voltage v(x,y) and a temporary vTemp(x,y):
    FieldType_t v(geom), vTemp(geom);

    // Start timing:
    Pooma::Clock clock;
    double start = clock.value();

    // Load initial condition v(x,y) = 0:
    v = 0.0;
    
    // Perform the Jacobi iteration. We apply the Jacobi formula twice
    // each loop:
    double error = 1000;
    int iteration = 0;
    while (error > 1e-6)
      {     
        iteration++;

        applyLaplacian(vTemp, v);
        applyLaplacian(v, vTemp);
        
        // The analytic solution is v(x, y) = 100 * x * y so we can test the
        // error:
        
        // Make sure calculations are done prior to scalar calculations.
        Pooma::blockAndEvaluate();
                 
        const double solution = v(center);
        const double analytic = 100.0 * v.x(center)(0) * v.x(center)(1);
        error = abs(solution - analytic);
        if (iteration % 1000 == 0)
          std::cout << "Iteration: " << iteration << "; "
              << "Error: " << error << std::endl;
      }

    std::cout << "Wall clock time: " << clock.value() - start << std::endl;
    std::cout << "Iteration: " << iteration << "; "
              << "Error: " << error << std::endl;

    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Laplace2.cpp,v $   $Author: richard $
// $Revision: 1.7 $   $Date: 2004/11/01 18:15:28 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
