// -*- C++ -*-
//
// Copyright (C) 2002  Jeffrey Oldham
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Global Function Templates:
//   product
//-----------------------------------------------------------------------------

#ifndef POOMA_HYDRODYNAMICS_PRODUCT_H
#define POOMA_HYDRODYNAMICS_PRODUCT_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Global Function Templates:
// 
// product()   :  mathematical vector operation
//            
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Global Function Templates:
// 
// product() : Perform a mathematical vector operation.  I do not know
// its name, but it is useful when computing the volume of areas.
//-----------------------------------------------------------------------------

// This Vector operation proves useful, but I do not know the name of
// the corresponding mathematical operation.
// TODO: Change to specialization for Dim=2.
template <int Dim, class T, class E>
Vector<Dim,T,E>
product(const Vector<Dim,T,E> &v, const Vector<Dim,T,E> &w)
{
  Vector<2,T,E> answer;
  answer(0) = v(0) * w(1);
  answer(1) = -v(1) * w(0);
  return answer;
}


//-----------------------------------------------------------------------------
//
// Full Description: product() versions that operate on `Field's.
// 
//-----------------------------------------------------------------------------

struct FnProduct
{
  PETE_EMPTY_CONSTRUCTORS(FnProduct)
  template<class T1, class T2>
  inline typename BinaryReturn<T1, T2, FnProduct >::Type_t
  operator()(const T1 &a, const T2 &b) const
  {
    return (product(a,b));
  }
};

template<class G1,class T1,class E1,int D2,class T2,class E2>
inline typename MakeReturn<BinaryNode<FnProduct,
  typename CreateLeaf<Field<G1,T1,E1> >::Leaf_t,
  typename CreateLeaf<Array<D2,T2,E2> >::Leaf_t> >::Expression_t
product(const Field<G1,T1,E1> & l,const Array<D2,T2,E2> & r)
{
  typedef BinaryNode<FnProduct,
    typename CreateLeaf<Field<G1,T1,E1> >::Leaf_t,
    typename CreateLeaf<Array<D2,T2,E2> >::Leaf_t> Tree_t;
  return MakeReturn<Tree_t>::make(Tree_t(
    CreateLeaf<Field<G1,T1,E1> >::make(l),
    CreateLeaf<Array<D2,T2,E2> >::make(r)));
}

template<class G1,class T1,class E1,class T2>
inline typename MakeReturn<BinaryNode<FnProduct,
  typename CreateLeaf<Field<G1,T1,E1> >::Leaf_t,
  typename CreateLeaf<T2 >::Leaf_t> >::Expression_t
product(const Field<G1,T1,E1> & l,const T2 & r)
{
  typedef BinaryNode<FnProduct,
    typename CreateLeaf<Field<G1,T1,E1> >::Leaf_t,
    typename CreateLeaf<T2 >::Leaf_t> Tree_t;
  return MakeReturn<Tree_t>::make(Tree_t(
    CreateLeaf<Field<G1,T1,E1> >::make(l),
    CreateLeaf<T2 >::make(r)));
}

template<int D1,class T1,class E1,class G2,class T2,class E2>
inline typename MakeReturn<BinaryNode<FnProduct,
  typename CreateLeaf<Array<D1,T1,E1> >::Leaf_t,
  typename CreateLeaf<Field<G2,T2,E2> >::Leaf_t> >::Expression_t
product(const Array<D1,T1,E1> & l,const Field<G2,T2,E2> & r)
{
  typedef BinaryNode<FnProduct,
    typename CreateLeaf<Array<D1,T1,E1> >::Leaf_t,
    typename CreateLeaf<Field<G2,T2,E2> >::Leaf_t> Tree_t;
  return MakeReturn<Tree_t>::make(Tree_t(
    CreateLeaf<Array<D1,T1,E1> >::make(l),
    CreateLeaf<Field<G2,T2,E2> >::make(r)));
}

template<class T1,class G2,class T2,class E2>
inline typename MakeReturn<BinaryNode<FnProduct,
  typename CreateLeaf<T1 >::Leaf_t,
  typename CreateLeaf<Field<G2,T2,E2> >::Leaf_t> >::Expression_t
product(const T1 & l,const Field<G2,T2,E2> & r)
{
  typedef BinaryNode<FnProduct,
    typename CreateLeaf<T1 >::Leaf_t,
    typename CreateLeaf<Field<G2,T2,E2> >::Leaf_t> Tree_t;
  return MakeReturn<Tree_t>::make(Tree_t(
    CreateLeaf<T1 >::make(l),
    CreateLeaf<Field<G2,T2,E2> >::make(r)));
}

#endif /* POOMA_HYDRODYNAMICS_PRODUCT_H */
