// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class:
//   ConstantFaceBC
//-----------------------------------------------------------------------------

#ifndef CONSTANTFACEBC_H
#define CONSTANTFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// ConstantVectorComponentBC: BCondCategory class setting the
//		   specified vector component in all guard layers beyond a
//                 specified (logically) rectilinear mesh face to a constant 
//                 value.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/NoInit.h"
#include "Utilities/PAssert.h"

#include "Field/Updater/Updater.h"
#include "Field/Updater/UpdaterBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag> class Field;


//-----------------------------------------------------------------------------
// Full Description:
// 
// ConstantVectorComponentBC is a specialization of UpdaterCategory.
// 
// It represents a boundary condition on a logically rectilinear
// domain where the value of one vector component on that face is a
// constant. The setConstant() function provides a means to have a
// time-dependent BC.  A constructor switch allows the BC to enforce
// that the mesh-boundary value is zero; this affects only
// vertex-centered Field values/components because the boundary is
// defined to be the last vertex.
// 
// The T template parameter is the type of the constant value.
//-----------------------------------------------------------------------------

template <class T>
class ConstantVectorComponentBC
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ConstantVectorComponentBC(int face, int vectorComponent,
			    const T &constant,
			    bool enforceConstantBoundary = false) 
  : face_m(face),
    vectorComponent_m(vectorComponent),
    constant_m(constant), 
    enforceConstantBoundary_m(enforceConstantBoundary)  
    { }
    
  ConstantVectorComponentBC(const ConstantVectorComponentBC &model) 
  : face_m(model.face_m), 
    vectorComponent_m(model.vectorComponent_m),
    constant_m(model.constant_m), 
    enforceConstantBoundary_m(model.enforceConstantBoundary_m)
    { }


  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ConstantVectorComponentBC &operator=(const ConstantVectorComponentBC &rhs)
  {
    face_m = rhs.face_m;
    vectorComponent_m = rhs.vectorComponent_m;
    constant_m = rhs.constant_m;
    enforceConstantBoundary_m = rhs.enforceConstantBoundary_m;

    return *this;
  }


  //---------------------------------------------------------------------------
  // Accessors/modifiers for data members.
  
  int face() const { return face_m; }
  int vectorComponent() const { return vectorComponent_m; }
  T constant() const { return constant_m; }
  bool enforceConstantBoundary() const { return enforceConstantBoundary_m; }

  // User may want to change the constant's value, e.g., for time-dependence.

  void setConstant(T newConstant) { constant_m = newConstant; }
  
  
private:

  int face_m;
  int vectorComponent_m;
  T constant_m;
  bool enforceConstantBoundary_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for ConstantVectorComponentBC
// ----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
class Updater<Field<GeometryTag, T, EngineTag>,
              ConstantVectorComponentBC<typename T::Element_t> > :
  public UpdaterBase<Field<GeometryTag, T, EngineTag>,
                     ConstantVectorComponentBC<typename T::Element_t> >
{
public:

  typedef Updater<Field<GeometryTag, T, EngineTag>,
                  ConstantVectorComponentBC<typename T::Element_t> > 
    This_t;
  typedef UpdaterBase<Field<GeometryTag, T, EngineTag>,
                      ConstantVectorComponentBC<typename T::Element_t> > 
    Base_t;
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  
  
  //---------------------------------------------------------------------------
  // Constructors.

  Updater(const Subject_t &s,
	  const ConstantVectorComponentBC<typename T::Element_t> &bc)
  : Base_t(s, bc),
    domain_m(Pooma::NoInit())
    {
      // This only makes sense if the subject has no sub-fields.
       
      PAssert(subject().numSubFields() == 0);
      
      // Set up the domain.
      
      domain_m = subject().totalDomain();
      
      // Get the direction.

      int d = bc.face() / 2;

      // The other directions span the subject's total domain. 
      // Therefore, we just chop out the guard layers, taking care to 
      // handle the case where we are enforcing a zero boundary 
      // (appropriate only for vert-centering).
      
      int adjust;
      if (bc.enforceConstantBoundary() && 
        subject().fieldEngine().offsets()[d] == 1) 
        adjust = 0;
      else
        adjust = 1;
  
      // Select the high or low face.

      if (bc.face() & 1) 
        {
          // High face.
 
          // Get the number of guard layers in the upper direction.

          int nGuards = subject().fieldEngine().guardLayers().upper(d);
          
          // Adjust the domain.
                   
          domain_m[d] = Interval<1>(domain_m[d].max() - nGuards + adjust, 
			                        domain_m[d].max());
        } 
      else 
        {
          // Low face.
        
          // Get the number of guard layers in the lower direction.

          int nGuards = subject().fieldEngine().guardLayers().lower(d);
          
          // Adjust the domain.
                   	
          domain_m[d] = Interval<1>(domain_m[d].min(), 
				    domain_m[d].min() + nGuards - adjust);
        }
    }


  //---------------------------------------------------------------------------
  // Destructor.

  ~Updater() { }


  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void update()
    {
      subject()(domain_m).comp(category().vectorComponent()) =
	category().constant();
    }
    

private:

  typename Subject_t::Domain_t domain_m;
};


//-----------------------------------------------------------------------------
// AllConstantVectorComponentBC is a functor class that sets all of a Field's BCs to be 
// ConstantVectorComponentBC.
//-----------------------------------------------------------------------------

template<class T>
class AllConstantVectorComponentBC
{
public:

  AllConstantVectorComponentBC(int vectorComponent, const T &constant,
			       bool enforceConstantBoundary = false)
  : vectorComponent_m(vectorComponent), const_m(constant),
    ecb_m(enforceConstantBoundary) { }
  
  template<class GeometryTag, class EngineTag>
  void operator()(Field<GeometryTag, T, EngineTag> &f) const
    {
      typedef Field<GeometryTag, T, EngineTag> Subject_t;
      
      for (int i = 0; i < 2 * Subject_t::dimensions; i++)
        f.addUpdater(ConstantVectorComponentBC<T>(i, vectorComponent_m,
						  const_m, ecb_m));
    }

private:

  int vectorComponent_m;
  T const_m;
  bool ecb_m;
};

#endif // CONSTANTFACEBC_H
