// -*- C++ -*-
//
// Copyright (C) 2002  Jeffrey Oldham
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes:
//   ComputeVolumes
// Global Function Templates:
//   computeVolumes
//-----------------------------------------------------------------------------

#ifndef POOMA_HYDRODYNAMICS_STENCIL_H
#define POOMA_HYDRODYNAMICS_STENCIL_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// ComputeVolumes  :  Volume-computation operator (functor) on vertex-
//		      centered Fields.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// 
// Global Function Templates:
// 
// computeVolumes() : Wrapper function around
//		      FieldStencil<ComputeVolumes>::operator ().  The
//		      ComputeVolumes functors actually used are
//		      partial specializations.
//            
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Field/Field.h"
#include "Field/DiffOps/FieldStencil.h"
#include "Product.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// ComputeVolumes:
//
// ComputeVolumes is a functor class serving as the "Functor" template
// parameter for FieldStencil<Functor,Expression>, which implements a
// discrete volume operator.
// Partial specializations implement vertex-centered input and
// cell-centered output.
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// 
// Global Function Templates:
// 
// computeVolumes() : Given a vertex-centered Field, return a
//		      ConstField with cell-centered scalar entries
//		      equalling the cells' volumes.
//
//-----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// General Total template
// ----------------------------------------------------------------------------

template<class T2, class Geometry>
class ComputeVolumes;


// ----------------------------------------------------------------------------
// Partial specializations of ComputeVolumes
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// ComputeVolumes Vector/Vert -> Scalar/Cell
// ----------------------------------------------------------------------------

template<class T2, class Geometry>
class ComputeVolumes;

template<class T2, int Dim, class TM, class CS>
class ComputeVolumes<Vector<Dim, T2>, UniformRectilinear<Dim, TM, CS> >
{
public:

  typedef T2   OutputElement_t;

  Cell outputCentering() const
  {
    return Cell();
  }

  Loc<Dim> inputOffsets() const
  {
    // input centering is Vert:
    return Loc<Dim>(1);
  }

  // 
  // Constructors.
  // 

  // default version is required by default stencil engine constructor.

  ComputeVolumes()  { }

  template<class FE>
  ComputeVolumes(const FE &fieldEngine) { }

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return f(i1+1) - f(i1+0);
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.5 * std::abs (sum (product
				(f(i1+1,i2+1)-f(i1+0,i2+0),
				 f(i1+0,i2+1)-f(i1+1,i2+0))));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    // Use triple scalar product of 
    // f(i1+1,i2,i3), f(i1,i2+1,i3), f(i1,i2,i3+1).
    return
      f(i1+1,i2,i3)(0) *
       (f(i1,i2+1,i3)(1)*f(i1,i2,i3+1)(1) -
	f(i1,i2+1,i3)(2)*f(i1,i2,i3+1)(1))
      -
      f(i1+1,i2,i3)(1) *
       (f(i1,i2+1,i3)(0)*f(i1,i2,i3+1)(2) -
	f(i1,i2+1,i3)(2)*f(i1,i2,i3+1)(0))
      +
      f(i1+1,i2,i3)(2) *
       (f(i1,i2+1,i3)(0)*f(i1,i2,i3+1)(1) -
	f(i1,i2+1,i3)(1)*f(i1,i2,i3+1)(0));
  }
};


// ----------------------------------------------------------------------------
// 
// Global Function Templates:
//
// ----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
typename
FieldStencilSimple<ComputeVolumes<T, Geometry>,
  Field<Geometry, T, EngineTag> >::Type_t
computeVolumes(const Field<Geometry, T, EngineTag> &f)
{
  typedef ComputeVolumes<T, Geometry> Total_t;
  typedef FieldStencilSimple<Total_t, Field<Geometry, T, EngineTag> > Ret_t;
  return Ret_t::make(Total_t(f.fieldEngine()), f);
}

#endif /* POOMA_HYDRODYNAMICS_STENCIL_H */
