// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
//   Doof2d Example
//-----------------------------------------------------------------------------

// include files

// Bring in Pooma Array machinery.
#include "Pooma/Arrays.h"

// Define stencil object.
#include "stencil9.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Main program.
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

  Inform out;

  // create arrays

  Array<2, double, MultiPatch<UniformTag, Remote<Brick> > > a, b;

  // set problem size

  int n = 2000;
  out << "Problem Size: " << n << std::endl;

  int i, niters = 20;

  // create array domain and resize arrays

  Interval<1> I(1,n), J(1,n);
  Interval<2> domain(I,J);

  // Create a 2 x 2 layout

  UniformGridPartition<2> partition(Loc<2>(2, 2), GuardLayers<2>(1),
				    GuardLayers<2>(1));

  UniformGridLayout<2> layout(domain, partition, DistributedTag());

  a.initialize(layout);
  b.initialize(layout);

  // get domains and constant for diffusion stencil

  const double fact = 1.0/9.0;

  // reset array element values

  a = b = 0.0;
  Pooma::blockAndEvaluate();
  a(niters,niters) = 1000.0;

  // Run 9pt doof2d without coefficients using expression

  out << "Diffusion using expression ..." << std::endl;
  out << "iter = 0, a_mid = " << a(niters,niters) << std::endl;
  for (i=1; i<=niters; ++i)
  {
    b(I,J) = fact * (a(I+1,J+1) + a(I+1,J  ) + a(I+1,J-1) +
                     a(I  ,J+1) + a(I  ,J  ) + a(I  ,J-1) +
                     a(I-1,J+1) + a(I-1,J  ) + a(I-1,J-1));
    a = b;
    Pooma::blockAndEvaluate();
    out << "iter = " << i << ", a_mid = " << a(niters,niters)
              << std::endl;
  }

  // reset array element values

  a = b = 0.0;
  Pooma::blockAndEvaluate();
  a(niters,niters) = 1000.0;

  // Run 9pt doof2d without coefficients using stencil

  out << "\nDiffusion using stencil ..." << std::endl;
  out << "iter = 0, a_mid = " << a(niters,niters) << std::endl;
  Stencil<DoofNinePt> stencil9;
  for (i=1; i<=niters; ++i)
  {
    b(I,J) = stencil9(a);
    a = b;
    Pooma::blockAndEvaluate();
    out << "iter = " << i << ", a_mid = " << a(niters,niters)
              << std::endl;
  }

  out << "All Done!" << std::endl;
  
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Doof2d.cpp,v $   $Author: richard $
// $Revision: 1.17 $   $Date: 2004/11/01 18:15:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
