// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Overview: 
// Classes needed by the generic Conjugate-Gradient solver to solve
// the problem:
//     -Laplace(x)=f,
//     f(z) = 1, 1/3<z1<2/3, 1/3<z2<2/3
//     f(z) = 0 otherwise
//     x = 0 on z1=0,z1=1,z2=0,z2=1
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SOLVERS_KRYLOV_PROBLEM1_H
#define POOMA_BENCHMARKS_SOLVERS_KRYLOV_PROBLEM1_H

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------

// Helper classes for Conjugate-Gradient solver

class Dot {
public:
  Dot() : n_m(0) {}
  Dot(int n) : n_m(n) {}

  template<class EngineTag>
  double operator()(const Array<2,double,EngineTag>& a,
		    const Array<2,double,EngineTag>& b) const
  {
    Interval<1> I(2,n_m-1),J(2,n_m-1);
    double result;

    result = sum(a(I,J)*b(I,J));

    return result;
  }
private:
  int n_m;
};

class MinusLaplace {
public:
  MinusLaplace(int n) : i(2,n-1),j(2,n-1),h2i((n-1)*(n-1)) {}
  template<class EngineTag>
  void operator()(const Array<2,double,EngineTag>& a,
		  Array<2,double,EngineTag>& b) const
  {
    b(i,j) = h2i*(4*a(i,j)-a(i-1,j)-a(i+1,j)-a(i,j-1)-a(i,j+1));
  }  
private:
  int h2i; // 1.0/dx^2
  Interval<1> i,j;
};

template<class EngineTag>
void setUpProblem1(Array<2,double,EngineTag>& a,int n)
{
  Interval<1> I((n+2)/3,(2*n+1)/3),J((n+2)/3,(2*n+1)/3);

  a = 0.0;
  a(I,J) = 1.0;

}

/*

void setUpProblem2(Array<2,double,Brick>& a,int n)
{
  int i, j;
  double x,y;
  double fx,fy;
  double h = 1.0/(n-1);

  for (j = 1; j <= n; j++) {
    for (i = 1; i <= n; i++) {
      x = (i-1)*h;
      y = (j-1)*h;
      if (x<0.25) {
	fx = 0.0;
      } else if (x<0.5) {
	fx = (x-0.25);
      } else if (x<0.75) {
	fx = (0.75-x);
      } else {
	fx = 0.0;
      }
      if (y<0.25) {
	fy = 0.0;
      } else if (y<0.5) {
	fy = (y-0.25);
      } else if (y<0.75) {
	fy = (0.75-y);
      } else {
	fy = 0.0;
      }
      a(i,j) = fx*fy;
    }
  }
}

*/

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_BENCHMARKS_SOLVERS_KRYLOV_PROBLEM1_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Problem1.h,v $   $Author: richard $
// $Revision: 1.10 $   $Date: 2004/11/01 18:15:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
