// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Function:
//  CGSolve - conjugate gradient solver
//-----------------------------------------------------------------------------
//
//  Templated conjugate gradient solver based on the templated solver in the
//  book: Templates for the Solution of Linear Systems by R. Barrett, M. Berry,
//  T.F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout, R. Pozo,
//  C. Romine, and H. van der Vorst.
//
//  The major differences between the current version and the one in the book:
//  1) no preconditioner
//  2) takes a dot product class that defines the dot product
//  3) takes a Constructor class that tells you how to initialize arrays.
//  4) uses dot(a,a) for norm(a)
//  5) matrix operation y=Ax is represented by A(x,y) instead of y=A(x)
//
//  CGSolve(A,x,b,dot,init,max_iter,tol);
//    -solves Ax=b for x
//    -iterates until max_iter iterations or until
//     dot(Ax-b,Ax-b) < tol*dot(b,b)
//    -A must be self-adjoint with respect to dot and positive definite.
//
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGSOLVE_H
#define POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGSOLVE_H

template<class Matrix,class V1,class V2,class Dot,class Constructor>
void
CGSolve(const Matrix &A,V1& x,const V2& b,
	const Dot &dot,const Constructor &constructor,
	int &max_iter,double &tol)
{
    double resid,r2,r2new;
    double alpha, beta;

    typedef typename Constructor::Type_t VC;

    VC &r = constructor();
    VC &d = constructor();
    VC &q = constructor();

    double normb = dot(b,b);

    // r = b-Lx
    q = 0.0;
    A(x,q);
    r = b-q;
    d = r;

    if (normb == 0.0)  normb = 1;
  
    r2 = dot(r,r);
    resid = r2 / normb;

    int i;
    for (i = 1; (i <= max_iter)&&(resid>tol); i++) {
      A(d,q);
      alpha = r2 / dot(d, q);
      x += alpha*d;
      r -= alpha*q;

      r2new = dot(r, r);
      beta = r2new/r2;
      d = r + beta*d;

      r2 = r2new;
      resid = r2 / normb;
    }
    tol = resid;
    max_iter = i;

    constructor.destroy(r);
    constructor.destroy(d);
    constructor.destroy(q);
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGSOLVE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CGSolve.h,v $   $Author: richard $
// $Revision: 1.7 $   $Date: 2004/11/01 18:15:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
