// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class declaration of a Conjugate-Gradient Poisson solver.
// Solves -Laplace(x)=f,
//     f(z) = 1, 1/3<z1<2/3, 1/3<z2<2/3
//     f(z) = 0 otherwise
//     x = 0 on z1=0,z1=1,z2=0,z2=1
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINP2_H
#define POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINP2_H

// include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"
#include "CGSolve.h"
#include "Problem1.h"


//-----------------------------------------------------------------------------
// CgAInP2 is a Pooma II implementation of the Conjugate gradient solver
//-----------------------------------------------------------------------------

template<bool MP>
class CGABase { };


template<>
class CGABase<false> : public Implementation
{
public:

  typedef Array<2,double,Brick> Array_t;
  
  // Constructor is an abstraction that is intended to act something like
  // the STL allocator class.  Unlike allocator, Constructor is intended
  // to allocate single objects and returns a reference to the object.

  class Constructor {
  public:
    typedef Array_t Type_t;

    Constructor() {}
    Constructor(int n,int np = 1)
      : domain_m(Interval<1>(1,n), Interval<1>(1,n))
    {}
 
    Array_t &operator()() const
    {
      return *(new Array_t(domain_m));
    }

    void destroy(Array_t& a) const
    {
      delete (&a);
    }
  private:
    Interval<2> domain_m;
  };

};


template<>
class CGABase<true> : public Implementation
{
public:
  
  typedef Array<2,double,MultiPatch<GridTag,Brick> > Array_t;
  
  class Constructor {
  public:
    typedef Array_t Type_t;

    Constructor() : layout_m(0)
    {
    }
    Constructor(int n,int np = 1)
      : domain_m(Interval<1>(1,n), Interval<1>(1,n))
    {
      // Create the block sizes.
    
      Loc<2> blocks(np, np);

      // Create the partitioner.
    
      GridPartition<2> partition(blocks);
  
      // Create the layout.
    
      layout_m = new GridLayout<2>(domain_m, partition, ReplicatedTag());
    }

   Constructor(const Constructor &model)
   : domain_m(model.domain_m), layout_m(new GridLayout<2>(*model.layout_m))
   {
   }

   Constructor &operator=(const Constructor &rhs)
   {
     PAssert(rhs.layout_m != NULL);
     domain_m = rhs.domain_m;
     layout_m = new GridLayout<2>(*rhs.layout_m);
     
     return *this;
   }
   
   ~Constructor()
   {
     delete layout_m;
   }
 
    Array_t &operator()() const
    {
      return *(new Array_t(*layout_m));
    }

    void destroy(Array_t& a) const
    {
      delete (&a);
    }

  private:

    Interval<2> domain_m;

    // Layout.

    GridLayout<2>* layout_m;
  };
};

template<bool MP>
class CgAInP2 : public CGABase<MP>
{
public:

  typedef typename CGABase<MP>::Constructor Constructor;
  typedef typename CGABase<MP>::Array_t Array_t;
  
  CgAInP2(int np = 1) : np_m(np) { }

  //---------------------------------------------------------------------------
  // We are a PoomaII implementation.

  const char* type() const
  {
    return CGABase<MP>::P2Type();
  }

  const char* qualification() const
  {
    if (MP)
    {
      return "MP";
    }
    else
    {
      return "";
    }
  }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n) {
    // Save the problem size. We need to make sure we have evenly sized blocks.
    
    n_m = n;

    // create Initializer and initialize Arrays
    constructor_m = Constructor(n_m,np_m);

    // create Dot product functor
    dot_m = Dot(n_m);
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run()
  {
    Array_t &f = constructor_m();
    Array_t &x = constructor_m();

    setUpProblem1(f,n_m);
    x = 0.0;

    MinusLaplace minusLaplace(n_m);

    double tol=1.0E-10;
    int    max_iter=10000;
    CGSolve(minusLaplace,x,f,dot_m,constructor_m,max_iter,tol);

    Pooma::blockAndEvaluate();

    // save result for checking
    check_m = x(n_m / 2, n_m / 2);
    // save number of iterations
    iters_m = max_iter;
    // tally up the flops
    flops_m = 11 * ((double)n_m - 2) * ((double)n_m - 2) +
              iters_m * (10 * ((double)n_m - 2) * ((double)n_m - 2) + 6 * (double)n_m * (double)n_m);

    constructor_m.destroy(f);
    constructor_m.destroy(x);
  }

  //---------------------------------------------------------------------------
  // Just run the setup

  void runSetup() {
    Array_t& f = constructor_m();
    Array_t& x = constructor_m();
    setUpProblem1(f,n_m);
    constructor_m.destroy(f);
    constructor_m.destroy(x);
  }

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const { return check_m; }
  
  //---------------------------------------------------------------------------
  // Returns the number of flops.

  double opCount() const { return flops_m; }

private:

  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;

  // Number of patches.

  int np_m;
  
  // Dot product.

  Dot dot_m;

  // Array initializer

  Constructor constructor_m;

  //---------------------------------------------------------------------------
  // Check value.
  
  double check_m;

  //---------------------------------------------------------------------------
  // Iteration count
  
  int iters_m;

  //---------------------------------------------------------------------------
  // Flop count
  
  int flops_m;
};

#endif // POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINP2_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CGAInP2.h,v $   $Author: richard $
// $Revision: 1.30 $   $Date: 2004/11/01 18:15:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
