// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class declaration of a Conjugate-Gradient Poisson solver (C version).
// Solves -Laplace(x)=f,
//     f(z) = 1, 1/3<z1<2/3, 1/3<z2<2/3
//     f(z) = 0 otherwise
//     x = 0 on z1=0,z1=1,z2=0,z2=1
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINC_H
#define POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINC_H

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"
#include <stdlib.h>

// function declarations

extern "C" {
void
runCGAInC(double* f, double* x, double* d, double* q, double* r,
          int n, int* iters);
}


//-----------------------------------------------------------------------------
// CgAInC is a C implementation of the Conjugate gradient solver
//-----------------------------------------------------------------------------

class CgAInC : public Implementation {
public:

  //---------------------------------------------------------------------------
  // Our default constructor just ensures we don't run into problems.

  CgAInC() {
    f_m = x_m = d_m = q_m = r_m = NULL;
  }
  
  //---------------------------------------------------------------------------
  // We need to delete the storage we've allocated.

  ~CgAInC() {
    delete [] f_m;
    delete [] x_m;
    delete [] d_m;
    delete [] q_m;
    delete [] r_m;
  }
  
  //---------------------------------------------------------------------------
  // We are a C implementation.

  const char* type() const { return CType(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n) {
    // delete and reallocate arrays

    delete [] f_m;
    delete [] x_m;
    delete [] d_m;
    delete [] q_m;
    delete [] r_m;

    f_m = new double[n * n];
    x_m = new double[n * n];
    d_m = new double[n * n];
    q_m = new double[n * n];
    r_m = new double[n * n];
    
    PInsist(f_m != NULL, "Memory allocation failure of f_m.");
    PInsist(x_m != NULL, "Memory allocation failure of x_m.");
    PInsist(d_m != NULL, "Memory allocation failure of d_m.");
    PInsist(q_m != NULL, "Memory allocation failure of q_m.");
    PInsist(r_m != NULL, "Memory allocation failure of r_m.");

    // save problem size    
    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run() {
    
    setInitialConditions();

    runCGAInC(f_m, x_m, d_m, q_m, r_m, n_m, &iters_m);

    // save result for checking
    check_m = x_m[n_m/2 - 1 + n_m * (n_m/2 - 1)];

    // tally up the flops
    flops_m = 11 * ((double)n_m - 2) * ((double)n_m - 2) +
              iters_m * (10 * ((double)n_m - 2) * ((double)n_m - 2) 
	      + 6 * (double)n_m * (double)n_m);
  }    

  //---------------------------------------------------------------------------
  // Just run the setup

  void runSetup() {
    setInitialConditions();
  }

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const { return check_m; }
  
  //---------------------------------------------------------------------------
  // Returns the number of flops.

  double opCount() const { return (double)flops_m; }

private:

  //---------------------------------------------------------------------------
  // Initialize our arrays.

  void setInitialConditions() {
    int i, j;

    for (j = 1; j <= n_m; j++) {
      for (i = 1; i <= n_m; i++) {
        if ((i>=(2+n_m)/3) && (i<=(2*n_m+1)/3) && 
	    (j>=(2+n_m)/3) && (j<=(2*n_m+1)/3))
	{
          f_m[i - 1 + n_m * (j - 1)] = 1.0;
        }
        else {
          f_m[i - 1 + n_m * (j - 1)] = 0.0;
        }
	x_m[i - 1 + n_m * (j - 1)] = 0.0;
	d_m[i - 1 + n_m * (j - 1)] = 0.0;
	q_m[i - 1 + n_m * (j - 1)] = 0.0;
	r_m[i - 1 + n_m * (j - 1)] = 0.0;
      }
    }
  }
  
  //---------------------------------------------------------------------------
  // Pointers.
  
  double *f_m, *x_m, *d_m, *q_m, *r_m;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  double check_m;

  //---------------------------------------------------------------------------
  // Iteration count.
  
  int iters_m;

  //---------------------------------------------------------------------------
  // Flop count.
  
  double flops_m;
};

#endif // POOMA_BENCHMARKS_SOLVERS_KRYLOV_CGAINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CGAInC.h,v $   $Author: richard $
// $Revision: 1.19 $   $Date: 2004/11/01 18:15:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
