// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINC_H
#define POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINC_H

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"
#include <stdlib.h>

// function declarations

extern "C" {
void 
runJacobiInC(double* x0, double* x1, double* b, int n, double tfinal,
             double theta, double phi, int* iters);
}


//-----------------------------------------------------------------------------
// JacobiInC is a C implementation of the Jacobi iterative solver
//-----------------------------------------------------------------------------

class JacobiInC : public Implementation {
public:

  // constructor: set all data pointers to null.  
  JacobiInC() : x0_m(NULL), x1_m(NULL), b_m(NULL) {}

  // destructor: delete any allocated memory
  ~JacobiInC() {
    delete [] x0_m;
    delete [] x1_m;
    delete []  b_m;
  }

  //---------------------------------------------------------------------------
  // We are a C implementation.

  const char* type() const { return CType(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n) {
    // delete and reallocate arrays
    delete [] x0_m;
    delete [] x1_m;
    delete []  b_m;
    x0_m = new double[n * n];
    x1_m = new double[n * n];
    b_m  = new double[n * n];

    PInsist(x0_m != NULL, "Memory allocation failure of x0_m.");
    PInsist(x1_m != NULL, "Memory allocation failure of x1_m.");
    PInsist( b_m != NULL, "Memory allocation failure of  b_m.");

    // save problem size
    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run() {
    // call C implementation
    runJacobiInC(x0_m, x1_m, b_m, n_m, tfinal_s, theta_s, phi_s, &iters_m);

    // get check value
    check_m = x0_m[n_m/2 - 1 + n_m * (n_m/2 - 1)];
    // tally up the flops
    flops_m = 11 * ((double)n_m - 2) * ((double)n_m - 2) * int(tfinal_s/(2*(0.1/((double)n_m-1)))) +
              iters_m * 41 * ((double)n_m - 2) * ((double)n_m - 2);
  }    

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const { return check_m; }
  
  //---------------------------------------------------------------------------
  // Returns the number of flops.

  double opCount() const { return flops_m; }

private:

  //---------------------------------------------------------------------------
  // Arrays.
  
  double *x0_m, *x1_m, *b_m;
  
  //---------------------------------------------------------------------------
  // Parameters.
  
  static const double tfinal_s, theta_s, phi_s;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  double check_m;

  //---------------------------------------------------------------------------
  // Iteration count
  
  int iters_m;

  //---------------------------------------------------------------------------
  // Flop count
  
  double flops_m;
};

// initialize static variables
const double JacobiInC::tfinal_s = 0.5;
const double JacobiInC::theta_s = 0.5;
const double JacobiInC::phi_s = 1.0;

#endif // POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: JacobiInC.h,v $   $Author: richard $
// $Revision: 1.16 $   $Date: 2004/11/01 18:15:17 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
