// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Classes atestInCwithout and atestInCwith.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SIMPLEARRAY_ATESTINC_H
#define POOMA_BENCHMARKS_SIMPLEARRAY_ATESTINC_H

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"
#include <stdlib.h>

// function declarations

extern "C" {
  void runatestInCwithout(double* w, double* x, double* y, double* z, int n);
  void runatestInCwith(double* w, double* x, double* y, double* z, int n);
};


//-----------------------------------------------------------------------------
// Class definitions.
//-----------------------------------------------------------------------------

class atestInCwithout : public Implementation {
public:

  // constructor: just set data pointers to null
  atestInCwithout() : w_m(NULL), x_m(NULL), y_m(NULL), z_m(NULL) {}

  // destructor: clean up allocated memory
  ~atestInCwithout() {
    delete [] w_m;
    delete [] x_m;
    delete [] y_m;
    delete [] z_m;
  }

  // This is a C benchmark.

  const char* type() const { return CType(); }
  const char* qualification() const { return "w/o restrict"; }

  void initialize(int n) {
    // delete and reallocate the arrays
    delete [] w_m;
    delete [] x_m;
    delete [] y_m;
    delete [] z_m;

    w_m = new double[n * n];
    x_m = new double[n * n];
    y_m = new double[n * n];
    z_m = new double[n * n];

    PInsist(w_m != NULL, "Memory allocation failure of w_m.");
    PInsist(x_m != NULL, "Memory allocation failure of x_m.");
    PInsist(y_m != NULL, "Memory allocation failure of y_m.");
    PInsist(z_m != NULL, "Memory allocation failure of z_m.");

    // Save problem size.
    n_m = n;

    // initialize the arrays.
    setInitialConditions();
  }

  // run benchmark
  void run() {
    // reset result array w_m
    for (int j = 0; j < n_m; j++)
      for (int i = 0; i < n_m; i++)
	w_m[i + n_m * j] = 1.1 * n_m;

    // computational kernel
    runatestInCwithout(w_m, x_m, y_m, z_m, n_m);

    // save result for checking
    check_m =  w_m[n_m/2 - 1 + n_m * (n_m/2 - 1)];
  }

  // just run setup
  void runSetup() {
    // reset result array w_m
    for (int j = 0; j < n_m; j++)
      for (int i = 0; i < n_m; i++)
	w_m[i + n_m * j] = 1.1 * n_m;
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (3 * (double)n_m * (double)n_m); }

private:

  void setInitialConditions()
  {
    for (int j = 0; j < n_m; j++) {
      for (int i = 0; i < n_m; i++) {
        x_m[i + n_m * j] = 2.2 * n_m;
        y_m[i + n_m * j] = 3.3 * n_m;
        z_m[i + n_m * j] = 4.4 * n_m;
      }
    }
  }

  // Data Arrays.

  double *w_m, *x_m, *y_m, *z_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};


class atestInCwith : public Implementation {
public:

  // constructor: just set data pointers to null
  atestInCwith() : w_m(NULL), x_m(NULL), y_m(NULL), z_m(NULL) {}

  // destructor: clean up allocated memory
  ~atestInCwith() {
    delete [] w_m;
    delete [] x_m;
    delete [] y_m;
    delete [] z_m;
  }

  // This is a C benchmark.

  const char* type() const { return CType(); }
  const char* qualification() const { return "w/ restrict"; }

  void initialize(int n) {
    // delete and reallocate the arrays
    delete [] w_m;
    delete [] x_m;
    delete [] y_m;
    delete [] z_m;

    w_m = new double[n * n];
    x_m = new double[n * n];
    y_m = new double[n * n];
    z_m = new double[n * n];

    PInsist(w_m != NULL, "Memory allocation failure of w_m.");
    PInsist(x_m != NULL, "Memory allocation failure of x_m.");
    PInsist(y_m != NULL, "Memory allocation failure of y_m.");
    PInsist(z_m != NULL, "Memory allocation failure of z_m.");

    // Save problem size.
    n_m = n;

    // Initialize the arrays.
    setInitialConditions();
  }

  // run benchyark
  void run() {
    // reset result array w_m
    for (int j = 0; j < n_m; j++)
      for (int i = 0; i < n_m; i++)
	w_m[i + n_m * j] = 1.1 * n_m;

    // computational kernel
    runatestInCwith(w_m, x_m, y_m, z_m, n_m);

    // save result for checking
    check_m =  w_m[n_m/2 - 1 + n_m * (n_m/2 - 1)];
  }

  // just run setup
  void runSetup() {
    // reset result array w_m
    for (int j = 0; j < n_m; j++)
      for (int i = 0; i < n_m; i++)
	w_m[i + n_m * j] = 1.1 * n_m;
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (3 * (double)n_m * (double)n_m); }

private:

  void setInitialConditions()
  {
    for (int j = 0; j < n_m; j++) {
      for (int i = 0; i < n_m; i++) {
        x_m[i + n_m * j] = 2.2 * n_m;
        y_m[i + n_m * j] = 3.3 * n_m;
        z_m[i + n_m * j] = 4.4 * n_m;
      }
    }
  }

  // Data Arrays.

  double *w_m, *x_m, *y_m, *z_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_DOOF2D_DOOF2DINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: atestInC.h,v $   $Author: richard $
// $Revision: 1.21 $   $Date: 2004/11/01 18:15:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
