// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
// ----------------------------------------------------------------------
// Messaging benchmark.
// ----------------------------------------------------------------------

// ----------------------------------------------------------------------
// include files
// ----------------------------------------------------------------------

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "Utilities/PAssert.h"
#include <iostream>

// Use preprocessor to include/exclude Tulip and/or MPI, so this can be
// compiled on systems not having them:

#ifdef POOMA_BENCHMARKS_MESSAGING_TULIP
#include "TulipCoreOneRound.h"
#include "tulip_core.h"
#endif // POOMA_BENCHMARKS_MESSAGING_TULIP

// CodeWarrior:
#ifdef __MWERKS__
#define POOMA_BENCHMARKS_MESSAGING_MPI
#define POOMA_BENCHMARKS_MESSAGING_MPISTUBS
#endif // __MWERKS__

#ifdef POOMA_BENCHMARKS_MESSAGING_MPI
#ifdef POOMA_BENCHMARKS_MESSAGING_MPISTUBS
#include "MPIStubs.h"
#else
#include <mpi.h>
#endif // POOMA_BENCHMARKS_MESSAGING_MPISTUBS
#include "MpiSpmd.h"
#endif // POOMA_BENCHMARKS_MESSAGING_MPI

// ----------------------------------------------------------------------
// dropArg
//
// Given a list of command line arguments, drop two of them starting
// at a given location. Shift the args to the right of the indicated
// ones down.
// ----------------------------------------------------------------------

void dropArgs(int arg, int drop, int &argc, char ** &argv)
{
  assert(arg+drop <= argc);
  for (int i=arg+drop; i<argc; ++i)
    argv[i-drop] = argv[i];
  argc -= drop;
}

// ----------------------------------------------------------------------
// messagingParameters
//
// Pull out the parameters we care about.
//
// dim_s: The spacial dimension
// patch_real_elems_s: Number of real elements in a patch in each dimension.
//                     This gets multiplied by the size that the benchmark
//                     class calculates to vary the size of the test.
// guards_s: Number of guard cells in each dimension.
//
// Ignore unrecognized arguments, and remove the ones recognized.
// ----------------------------------------------------------------------

static int dim_s;
static int patch_real_elems_s;
static int guards_s;
static int *tiling_s;
static long iterations_s;
static bool mot_s; // Flags whether to exclude buffer-copy from timing.

void messagingParameters(int &argc, char ** &argv)
{
  // Defaults:
  dim_s = 1;
  patch_real_elems_s = 1;
  guards_s = 1;
  tiling_s = new int[dim_s];
  tiling_s[0] = 1;
  iterations_s = 10000;
  mot_s = false;

  int arg = 1;
  while (arg < argc)
    {
      if ( strcmp(argv[arg],"--dim") == 0 )
	{
	  assert(arg+1 < argc);
	  dim_s = atoi( argv[arg+1] );
	  dropArgs(arg,2,argc,argv);
	}
      else if ( strcmp(argv[arg],"--guards") == 0 )
	{
	  assert(arg+1 < argc);
	  guards_s = atoi( argv[arg+1] );
	  dropArgs(arg,2,argc,argv);
	}
      else if ( strcmp(argv[arg],"--elems") == 0 )
	{
	  assert(arg+1 < argc);
	  patch_real_elems_s = atoi( argv[arg+1] );
	  dropArgs(arg,2,argc,argv);
	}
      else if ( strcmp(argv[arg],"--iterations") == 0 )
	{
	  assert(arg+1 < argc);
	  iterations_s = atoi( argv[arg+1] );
	  dropArgs(arg,2,argc,argv);
	}
      else if ( strcmp(argv[arg],"--mot") == 0 )
	{
	  assert(arg+1 < argc);
	  mot_s = atoi( argv[arg+1] );
	  dropArgs(arg,2,argc,argv);
	}
      else if ( strcmp(argv[arg],"--tiling") == 0 )
	{
	  assert(dim_s>0);
	  assert(arg+dim_s < argc);
          delete [] tiling_s;
	  tiling_s = new int[dim_s];
	  for (int i=0; i<dim_s; ++i)
	    tiling_s[i] = atoi( argv[arg+1+i] );
	  dropArgs(arg,1+dim_s,argc,argv);
	}
      else 
        {
          arg++;
        }
    }
}

// ----------------------------------------------------------------------
// Main Program.
// ----------------------------------------------------------------------

int main(int argc, char **argv)
{
  //
  // Initialize the parallelism.
  //

  bool mpiInitialized = false;

#ifdef POOMA_BENCHMARKS_MESSAGING_TULIP
  tulip_core_setup(&argc,&argv);
#endif // POOMA_BENCHMARKS_MESSAGING_TULIP

#ifdef POOMA_BENCHMARKS_MESSAGING_MPI
  if (!mpiInitialized) {
    MPI_Init(&argc, &argv);
    mpiInitialized = true;
  }
#endif // POOMA_BENCHMARKS_MESSAGING_MPI

  //
  // Pull out the parameters specific to these tests.
  //
  messagingParameters(argc,argv);

#if defined(POOMA_BENCHMARKS_MESSAGING_TULIP) || defined(POOMA_BENCHMARKS_MESSAGING_MPI)
  //
  // Build a benchmark object to hold the benchmarks we will do.
  //
  Benchmark messaging(argc,argv,"Border exchange messaging");
#endif // POOMA_BENCHMARKS_MESSAGING_TULIP || POOMA_BENCHMARKS_MESSAGING_MPI

#ifdef POOMA_BENCHMARKS_MESSAGING_TULIP

  //
  // Add the Tulip tests we have.
  //
  messaging.addImplementation(new TulipCoreOneRound(dim_s,
						    tiling_s,
						    guards_s));

#endif // POOMA_BENCHMARKS_MESSAGING_TULIP

#ifdef POOMA_BENCHMARKS_MESSAGING_MPI

  // Add MPI tests:
  switch (dim_s) {

  case 1:
    messaging.addImplementation(new MpiSpmd<1>(tiling_s, guards_s, mot_s));
    break;

  case 2:
    messaging.addImplementation(new MpiSpmd<2>(tiling_s, guards_s, mot_s));
    break;

  case 3:
    messaging.addImplementation(new MpiSpmd<3>(tiling_s, guards_s, mot_s));
    break;

  default:
    // Puke and die on unimplemented Dim values:
    if (dim_s > 3) {
      std::cout << "Specialization for MpiSpmd<" << dim_s 
                << " not implemented; quit." << std::endl;
      MPI_Finalize();
      PInsist(false, "...quitting by calling PInsist.");
    }
    break;

  }

  // Set the Inform output object to be the one I like:
  int pe;
  MPI_Comm_rank(MPI_COMM_WORLD, &pe);
  int npe;
  MPI_Comm_size(MPI_COMM_WORLD, &npe);
  Inform* mpiInform = new Inform(NULL, 0, pe, npe);
  messaging.setInform(mpiInform);

  *mpiInform << npe << " MPI processes; ";
  *mpiInform << "Pooma::Clock::highSpeed = ";
  if (Pooma::Clock::highSpeed) {
    *mpiInform << "true";
  } else {
    *mpiInform << "false";
  }
  *mpiInform << std::endl;

#endif // POOMA_BENCHMARKS_MESSAGING_MPI

#if defined(POOMA_BENCHMARKS_MESSAGING_TULIP) || defined(POOMA_BENCHMARKS_MESSAGING_MPI)
  //
  // Do the benchmark:
  //
  messaging.setSamplingParameters(patch_real_elems_s,0,1);
  messaging.setIterations(iterations_s);
  messaging.run();
#endif // POOMA_BENCHMARKS_MESSAGING_TULIP || POOMA_BENCHMARKS_MESSAGING_MPI

#ifdef POOMA_BENCHMARKS_MESSAGING_MPI
  *mpiInform << "Finished messaging.run\n";
#else
  std::cout << "Finished messaging.run\n";
#endif // POOMA_BENCHMARKS_MESSAGING_MPI

  //
  // Close down the messaging.
  //

#ifdef POOMA_BENCHMARKS_MESSAGING_TULIP
  tulip_core_finalize();
#endif // POOMA_BENCHMARKS_MESSAGING_TULIP

#ifdef POOMA_BENCHMARKS_MESSAGING_MPI
  if (mpiInitialized) MPI_Finalize();
#endif // POOMA_BENCHMARKS_MESSAGING_MPI

}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MessageMain.cpp,v $   $Author: richard $
// $Revision: 1.11 $   $Date: 2004/11/01 18:15:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
