// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Classes GKPoissonInCppTran
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINCPPTRAN_H
#define POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINCPPTRAN_H

//-----------------------------------------------------------------------------
// include files
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"

#include <stdlib.h>


//-----------------------------------------------------------------------------
// GKPoissonInCppTran class definition.
//-----------------------------------------------------------------------------

class GKPoissonInCppTran : public Implementation {
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<2,double,Brick> Array2D;

  // Constructor:

  GKPoissonInCppTran(double gamma)
    : gamma0_m(gamma)
  {
    ijfact_m = 0.25 - gamma;
    norm_m   = 1.0 / (2.0 - gamma);
  }

  // This is a CppTran benchmark

  const char* type() const { return CppTranType(); }

  void initialize(int n) 
  {
    // Field domain for density and potential.

    Interval<1> N(n);
    Interval<2> fieldDomain(N, N);

    // Domain for array with gaurd cells. 

    Interval<1> NGC(-2,n+1);
    Interval<2> gcDomain(NGC,NGC);

    // Reset the size of member Arrays.

    phihat_m.initialize(fieldDomain);
    density_m.initialize(fieldDomain);

    phi_m.initialize(gcDomain);

    // Save the problem size.

    n_m = n;

    // The rest of the code is scalar. Put this here just
    // to make sure there are no outstanding data parallel
    // iterates for our data.

    Pooma::blockAndEvaluate();

    // Intialize Array element values.

    setInitialConditions();
  }

  void run() 
  {
    int i, j, iter;

    setInitialConditions();

    for (iter = 0; iter < 5; ++iter)
    {
      for (j = 0; j < n_m; ++j)
      {
        for (i = 0; i < n_m; ++i)
	      {
          phihat_m(i,j) = density_m(i,j) 
            + ijfact_m * phi_m(i,j)
            + 0.125  * ( phi_m(i+1,j+1) + phi_m(i+1,j-1) +
                         phi_m(i-1,j+1) + phi_m(i-1,j-1) )
            + 0.0625 * ( phi_m(i  ,j+2) + phi_m(i  ,j-2) +
                         phi_m(i+2,j  ) + phi_m(i-2,j  ) );
        }
      }

      for (j = 0; j < n_m; ++j)
      {
        for (i = 0; i < n_m; ++i)
        {
          phi_m(i,j) = norm_m * phihat_m(i,j);
        }
      }
    }

    // Save results for checking

    check_m = 0.0;
    for (j = 0; j < n_m; j++)
      check_m += phi_m(n_m / 2, j);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (66 * (double)n_m * (double)n_m); }

private:

  void setInitialConditions()
  {
    // Intialize Array element values.

    int i, j;

    for (j = 0; j < n_m; ++j) 
      {
	for (i = 0; i < n_m; ++i) 
	  {
	    phihat_m(i,j) = 0.0;
	    density_m(i,j) = 0.0;
	  }
      }

    for (j = -2; j < n_m + 2; ++j)
      {
	for (i = -2; i < n_m + 2; ++i)
	  {
	    phi_m(i,j) = 0.0;
	  }
      }

    density_m(n_m/2,n_m/2) = 1000.;

    phi_m(n_m/2,n_m/2) = 1000. * gamma0_m;
  }

  // 2D Data Arrays.

  Array2D phihat_m, phi_m, density_m;

  // Constants

  double gamma0_m, ijfact_m, norm_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINCPPTRAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: GKPoissonInCppTran.h,v $   $Author: richard $
// $Revision: 1.19 $   $Date: 2004/11/01 18:15:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
