// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class GKPoissonInC.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINC_H
#define POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINC_H

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"

#include <stdlib.h>

//-----------------------------------------------------------------------------
// Function declarations
//-----------------------------------------------------------------------------

extern "C" {
  void runGKPoissonInC(double* phi, double* density, double *phihat,
		       double gamma0, int n);
  void setInitialConditionsInC(double* phi, double* density, double *phihat,
			       double gamma0, int n);
};


//-----------------------------------------------------------------------------
// Class definitions.
//-----------------------------------------------------------------------------

class GKPoissonInC : public Implementation 
{
public:

  // Constructor: just set data pointers to null

  GKPoissonInC(double gamma0) 
    : phihat_m(0), density_m(0), phi_m(0), gamma0_m(gamma0) 
  { }
  
  // Destructor: clean up allocated memory

  ~GKPoissonInC() 
  {
    delete [] phihat_m;
    delete [] density_m;
    delete [] phi_m;
  }

  // This is a C benchmark.

  const char* type() const { return CType(); }

  void initialize(int n) 
  {
    // Delete and reallocate the arrays

    delete [] phihat_m;
    delete [] density_m;
    delete [] phi_m;

    // Save problem size.

    n_m = n;
    nmax_m = n + 4;

    // Size, with guard cells (which we put on
    // all arrays to make them aligned).

    const int size = nmax_m * nmax_m;

    phihat_m  = new double[size];
    density_m = new double[size];
    phi_m     = new double[size];

    PInsist(phihat_m  != NULL, "Memory allocation failure of phihat_m.");
    PInsist(density_m != NULL, "Memory allocation failure of density_m.");
    PInsist(phi_m     != NULL, "Memory allocation failure of phi_m.");

    // Initialize arrays

    setInitialConditions();
  }

  void run() 
  {
    // Run kernel

    runGKPoissonInC(phi_m, density_m, phihat_m, gamma0_m, n_m);

    // Save result

    Pooma::blockAndEvaluate();
    int nd = n_m / 2;
    check_m = 0.0;
    for (int j = 0; j < n_m; j++)
      check_m += phi_m[ nd + 2 + nmax_m * (j + 2) ];
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (66 * (double)n_m * (double)n_m); }

private:

  void setInitialConditions() 
  {
    setInitialConditionsInC(phi_m,phihat_m,density_m,gamma0_m,n_m);
  }

  // Data Arrays.

  double *phihat_m, *phi_m, *density_m;

  // Constants

  double gamma0_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m, nmax_m;

};

#endif // POOMA_BENCHMARKS_GKPOISSON_GKPOISSONINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: GKPoissonInC.h,v $   $Author: richard $
// $Revision: 1.17 $   $Date: 2004/11/01 18:15:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
