// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes Doof3dInP2UnOpt and Doof3dInP2Op.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_DOOF3D_DOOF3DINP2_H
#define POOMA_BENCHMARKS_DOOF3D_DOOF3DINP2_H

// include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"

#include <stdlib.h>


//-----------------------------------------------------------------------------
// Doof3dInP2UnOpt class definition.
//-----------------------------------------------------------------------------

class Doof3dInP2UnOpt : public Implementation
{
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<3,double,Brick> Array3D;

  // This is a Pooma II benchmark

  const char *type() const { return P2Type(); }
  const char *qualification() const { return "NoOpt"; }

  void initialize(int n) {
    // get new array domain
    Interval<1> N(1, n);
    Interval<3> newDomain(N, N, N);

    // Reset the size of member Arrays.
    a_m.initialize(newDomain);
    b_m.initialize(newDomain);

    // reset Intervals for diffusion stencil
    I = Interval<1>(2,n-1);
    J = Interval<1>(2,n-1);
    K = Interval<1>(2,n-1);

    // Save the problem size.
    n_m = n;

    // Intialize Array element values.
    a_m = 1.1 * n;
    b_m = 0.0;
  }

  // run benchmark
  void run() {
    // run kernel
    b_m(I,J,K) = fact_s * (
      a_m(I-1,J-1,K-1) + a_m(I,  J-1,K-1) + a_m(I+1,J-1,K-1) +
      a_m(I-1,J,  K-1) + a_m(I,  J,  K-1) + a_m(I+1,J,  K-1) +
      a_m(I-1,J+1,K-1) + a_m(I,  J+1,K-1) + a_m(I+1,J+1,K-1) +
      a_m(I-1,J-1,K  ) + a_m(I,  J-1,K  ) + a_m(I+1,J-1,K  ) +
      a_m(I-1,J,  K  ) + a_m(I,  J,  K  ) + a_m(I+1,J,  K  ) +
      a_m(I-1,J+1,K  ) + a_m(I,  J+1,K  ) + a_m(I+1,J+1,K  ) +
      a_m(I-1,J-1,K+1) + a_m(I,  J-1,K+1) + a_m(I+1,J-1,K+1) +
      a_m(I-1,J,  K+1) + a_m(I,  J,  K+1) + a_m(I+1,J,  K+1) +
      a_m(I-1,J+1,K+1) + a_m(I,  J+1,K+1) + a_m(I+1,J+1,K+1));

    Pooma::blockAndEvaluate();

    // save results for checking
    check_m = b_m(n_m / 2, n_m / 2, n_m / 2);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const {
    return ( 27 * (n_m - 2) * (n_m - 2) * (n_m - 2) );
  }

private:

  // 3D Data Arrays.

  Array3D a_m, b_m;

  // 1D Intervals

  Interval<1> I, J, K;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

  // constant factor

  static const double fact_s;
};

// static initializer
const double Doof3dInP2UnOpt::fact_s = 1.0 / 9.0;


//-----------------------------------------------------------------------------
// Stencil Doof27Pt
//-----------------------------------------------------------------------------

class Doof27Pt
{
public:
  Doof27Pt() {}

  template <class A>
  inline
  typename A::Element_t
  operator()(const A& a, int i, int j, int k) const
  {
    return ( fact_s * (
                a(i-1,j-1,k-1) + a(i,  j-1,k-1) + a(i+1,j-1,k-1) +
                a(i-1,j,  k-1) + a(i,  j,  k-1) + a(i+1,j,  k-1) +
                a(i-1,j+1,k-1) + a(i,  j+1,k-1) + a(i+1,j+1,k-1) +
                a(i-1,j-1,k  ) + a(i,  j-1,k  ) + a(i+1,j-1,k  ) +
                a(i-1,j,  k  ) + a(i,  j,  k  ) + a(i+1,j,  k  ) +
                a(i-1,j+1,k  ) + a(i,  j+1,k  ) + a(i+1,j+1,k  ) +
                a(i-1,j-1,k+1) + a(i,  j-1,k+1) + a(i+1,j-1,k+1) +
                a(i-1,j,  k+1) + a(i,  j,  k+1) + a(i+1,j,  k+1) +
                a(i-1,j+1,k+1) + a(i,  j+1,k+1) + a(i+1,j+1,k+1)) );
  }

  inline int lowerExtent(int) const { return 1; }
  inline int upperExtent(int) const { return 1; }

private:
  static const double fact_s;
};

const double Doof27Pt::fact_s = 1.0/9.0;


//-----------------------------------------------------------------------------
// Doof3dInP2Opt class definition.
//-----------------------------------------------------------------------------

class Doof3dInP2Opt : public Implementation {
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<3,double,Brick> Array3D;

  // This is a Pooma II benchmark

  const char *type() const { return P2Type(); }
  const char *qualification() const { return "Opt"; }

  void initialize(int n) {
    // get new array domain
    Interval<1> N(1, n);
    Interval<3> newDomain(N, N, N);

    // Reset the size of member Arrays.
    a_m.initialize(newDomain);
    b_m.initialize(newDomain);

    // reset Intervals for diffusion stencil
    I = Interval<1>(2,n-1);
    J = Interval<1>(2,n-1);
    K = Interval<1>(2,n-1);

    // Save the problem size.
    n_m = n;

    // Initialize Array element values.
    a_m = 1.1 * n;
    b_m = 0.0;
  }

  // run benchmark
  void run() {
    // apply diffusion stencil
    b_m(I,J, K) = stencil_m( a_m );

    Pooma::blockAndEvaluate();

    // save results for checking
    check_m = b_m(n_m/2,n_m/2,n_m/2);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const
  {
    return ( 27 * ((double)n_m - 2) * ((double)n_m - 2) * ((double)n_m - 2) );
  }

private:

  // 3D Data Arrays.

  Array3D a_m, b_m;

  // 1D Intervals

  Interval<1> I, J, K;

  // Stencil object

  Stencil<Doof27Pt> stencil_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};


#endif // POOMA_BENCHMARKS_DOOF3D_DOOF3DINP2_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Doof3dInP2.h,v $   $Author: richard $
// $Revision: 1.18 $   $Date: 2004/11/01 18:15:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
