// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class Doof2dInC.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_DOOF3D_DOOF3DINC_H
#define POOMA_BENCHMARKS_DOOF3D_DOOF3DINC_H

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"

#include <stdlib.h>

// function declarations

extern "C" {
  void runDoof3dInC(double* a, double* b, int n);
  void runDoof3dInCRestrict(double* a, double* b, int n);
};


//-----------------------------------------------------------------------------
// Class definitions.
//-----------------------------------------------------------------------------

template<bool UseRestrict>
class Doof3dInC : public Implementation {
public:

  // constructor: just set data pointers to null
  Doof3dInC() : a_m(NULL), b_m(NULL) {}

  // destructor: clean up allocated memory
  ~Doof3dInC() {
    delete [] a_m;
    delete [] b_m;
  }

  // This is a C benchmark.

  const char *type() const { return CType(); }
  const char *qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n) {
    // delete and reallocate the arrays
    delete [] a_m;
    delete [] b_m;

    a_m = new double[n * n * n];
    b_m = new double[n * n * n];

    PInsist(a_m != NULL, "Memory allocation failure of a_m.");
    PInsist(b_m != NULL, "Memory allocation failure of b_m.");

    // Save problem size.
    n_m = n;

    // initialize arrays
    setInitialConditions();
  }

  void run() {
    Pooma::blockAndEvaluate();
    
    // run kernel
    if (UseRestrict)
      runDoof3dInCRestrict(a_m, b_m, n_m);
    else
      runDoof3dInC(a_m, b_m, n_m);

    // save result
    check_m =  b_m[n_m / 2 - 1 + n_m * (n_m / 2 - 1 + n_m * (n_m / 2 - 1))];
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const
  {
    return ( 27 * (n_m - 2) * (n_m - 2) * (n_m - 2) );
  }

private:

  void setInitialConditions() 
  {
    Pooma::blockAndEvaluate();
    
    // initialize the arrays.
    for (int k = 0; k < n_m; k++) {
      for (int j = 0; j < n_m; j++) {
        for (int i = 0; i < n_m; i++) {
	  a_m[i + n_m * (j + n_m * k)] = 1.1 * n_m;
          b_m[i + n_m * (j + n_m * k)] = 0.0;
        }
      }
    }
  }

  // Data Arrays.

  double *a_m, *b_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_DOOF3D_DOOF3DINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Doof3dInC.h,v $   $Author: richard $
// $Revision: 1.12 $   $Date: 2004/11/01 18:15:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
