// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_DOOF2D_DOOF2DINC_H
#define POOMA_BENCHMARKS_DOOF2D_DOOF2DINC_H

//-----------------------------------------------------------------------------
// Class Doof2dInC.
//-----------------------------------------------------------------------------

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"

#include <stdlib.h>

// function declarations

extern "C" {
  void runDoof2dSetup(double* a, double* b, int n);
  void runDoof2dInC(double* a, double* b, int n);
  void runDoof2dCoefsInC(double* a, double* b, double* c, int n);
  void runDoof2dInCRestrict(double* a, double* b,int n);
  void runDoof2dCoefsInCRestrict(double* a, double* b, double* c, int n);
}


//-----------------------------------------------------------------------------
// Class definitions.
//-----------------------------------------------------------------------------

template<bool UseRestrict>
class Doof2dInC : public Implementation {
public:

  // constructor: just set data pointers to null
  Doof2dInC() : a_m(NULL), b_m(NULL) {}

  // destructor: clean up allocated memory
  ~Doof2dInC() {
    delete [] a_m;
    delete [] b_m;
  }

  // This is a C benchmark.

  const char *type() const { return CType(); }
  const char *qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n) {
    // delete and reallocate the arrays
    delete [] a_m;
    delete [] b_m;

    a_m = new double[n * n];
    b_m = new double[n * n];

    PInsist(a_m != NULL, "Memory allocation failure of a_m.");
    PInsist(b_m != NULL, "Memory allocation failure of b_m.");

    // Save problem size.
    n_m = n;

    // Ensure that all memory is paged in.

    for (int j = 0; j < n_m; j++) {
      for (int i = 0; i < n_m; i++) {
	a_m[i + n_m * j] = 0.0;
        b_m[i + n_m * j] = 0.0;
      }
    }
  }

  void run() {
    // run kernel
    
    if (UseRestrict)
      runDoof2dInCRestrict(a_m, b_m, n_m);
    else
      runDoof2dInC(a_m, b_m, n_m);

    // save result
    check_m =  b_m[n_m / 2 - 1 + n_m * (n_m / 2 - 1)];
  }

  void runSetup()
  {
    runDoof2dSetup(a_m, b_m, n_m);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 90 * ((double)n_m - 2) * ((double)n_m - 2) ); }

private:

  // Data Arrays.

  double *a_m, *b_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

template<bool UseRestrict>
class Doof2dCoefsInC : public Implementation {
public:

  Doof2dCoefsInC() : a_m(NULL), b_m(NULL) {

    // Initialize rand() random number generator.

    srand(12345U);

    // Loop over array elements and assign random values.

    for (int i = 0; i < 9; i++)
      c_m[i] = rand() / (static_cast<double>(RAND_MAX)+1);
  }

  // This is a C benchmark.

  const char *type() const { return CType(); }
  const char *qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n) {
    // delete and reallocate arrays
    delete [] a_m;
    delete [] b_m;

    a_m = new double[n * n];
    b_m = new double[n * n];

    PInsist(a_m != NULL, "Memory allocation failure of a_m.");
    PInsist(b_m != NULL, "Memory allocation failure of b_m.");

    // Save problem size.
    n_m = n;

    // Ensure that all memory is paged in.

    for (int j = 0; j < n_m; j++) {
      for (int i = 0; i < n_m; i++) {
	a_m[i + n_m * j] = 0.0;
        b_m[i + n_m * j] = 0.0;
      }
    }
  }

  void run() {
    // run kernel
    if (UseRestrict)
      runDoof2dCoefsInCRestrict(a_m, b_m, c_m, n_m);
    else
      runDoof2dCoefsInC(a_m, b_m, c_m, n_m);

    // save result for checking
    check_m =  b_m[n_m / 2 - 1 + n_m * (n_m / 2 - 1)];
  }

  void runSetup()
  {
    runDoof2dSetup(a_m, b_m, n_m);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 180 * ((double)n_m - 2) * ((double)n_m - 2) ); }

private:

  // Data Arrays.

  double *a_m, *b_m;

  // Coefficient Array

  double c_m[9];

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_DOOF2D_DOOF2DINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Doof2dInC.h,v $   $Author: richard $
// $Revision: 1.26 $   $Date: 2004/11/01 18:15:11 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
