// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class Loop18InCppTran.  This class executes Blitz++ benchmarking loop 18
// using the Pooma II Array class and elementwise evaluation in a for loop.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INCPPTRAN_H
#define POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INCPPTRAN_H

// include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"


//-----------------------------------------------------------------------------
// Class definition.
//-----------------------------------------------------------------------------

class Loop18InCppTran : public Implementation {

public:

  // typedef for the Array type we'll be using here.
  typedef Array<1, double, Brick> DoubleArray1D;

  // this is a C++-tran benchmark
  const char* type(void) const { return CppTranType(); }

  void initialize(int n) {
    // create new 1D Interval from 1 to n
    Interval<1> newDomain(1,n);

    // reset size of member Arrays
    x_m.initialize(newDomain);
    a_m.initialize(newDomain);
    b_m.initialize(newDomain);

    // save problem size
    n_m = n;

    // intialize Array element values
    setInitialConditions();

    return;
  }

  // run benchmark
  void run(void) {
    // evaluate Array expression for each element using a loop
    for (int i=1; i<=n_m; ++i)
      x_m(i) = (u_m * a_m(i)) + (v_m * b_m(i));

    // store middle element value for checking result
    check_m = x_m(n_m/2);

    return;
  }

  // return value for checking result of benchmark run
  double resultCheck(void) const { return check_m; }

  // return flop count for this kernel
  double opCount(void) const { return (3 * (double)n_m); }

private:

  void setInitialConditions(void) {
    // initialize rand() random number generator
    srand(12345U);

    // loop over Array elements and assign random values
    for (int i=1; i<=n_m; ++i) {
      a_m(i) = rand()/(static_cast<double>(RAND_MAX)+1);
      b_m(i) = rand()/(static_cast<double>(RAND_MAX)+1);
    }

    return;
  }

  // 1D Data Arrays
  DoubleArray1D x_m, a_m, b_m;

  // Problem check value
  double check_m;

  // Problem Size
  int n_m;

  // Scalar constants
  static const double u_m;
  static const double v_m;

};


// Assign static constants
const double Loop18InCppTran::u_m = 0.39123982498157938742;
const double Loop18InCppTran::v_m = 0.39123982498157938742;


#endif // POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INCPPTRAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Loop18InCppTran.h,v $   $Author: richard $
// $Revision: 1.22 $   $Date: 2004/11/01 18:15:04 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
