// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes AbcCppTran, AbcP2, AbcNinePt, AbcOpt
// Implementation Classes AbcStorage, AbcBase, 
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_ABC_H
#define POOMA_BENCHMARKS_ABC_H

// Include files

#include "Pooma/Arrays.h"
#include "Pooma/Fields.h"
#include "Utilities/Benchmark.h"

#include <cstdlib>
#include <typeinfo>
#include <string>

#if POOMA_CHEETAH
  typedef DistributedTag LayoutTag_t;
#else
  typedef ReplicatedTag LayoutTag_t;
#endif

//-----------------------------------------------------------------------------
// PatchString class definitions.  These return a string notation of
// the engine type.
//-----------------------------------------------------------------------------

template<class ETag>
struct PatchString
{
  static
  std::string get()
  {
    if (typeid(ETag) == typeid(Brick))
      return "Bk";
    else
      return "CmBk";
  }
};

template<class ETag>
struct PatchString<Remote<ETag> >
{
  static
  std::string get()
  {
    return "r" + PatchString<ETag>::get();
  }
};

template<class LTag, class ETag>
struct PatchString<MultiPatch<LTag, ETag> >
{
  static
  std::string get()
  {
    return PatchString<ETag>::get() + "MP";
  }
};

template<int D, class T, class ETag>
std::string qualification(const Array<D, T, ETag> &)
{
  return PatchString<ETag>::get();
}

template<class Mesh, class T, class ETag>
std::string qualification(const Field<Mesh, T, ETag> &)
{
  return "Fd" + PatchString<ETag>::get();
}

//-----------------------------------------------------------------------------
// AbcStorage class definitions.  These classes allocate the array
// or field and initialize it.  We define this as a partially
// specialized class so it is easy to make subclasses work for Uniform
// MultiPatch (UMP) or not.
//-----------------------------------------------------------------------------

template<class Storage, bool UMP>
class AbcStorage
{
};

template<class ETag>
class AbcStorage<Array<2, double, ETag>, false>
{
public:

  void initializeStorage(int &n, int np, int ng) 
  {
    // Get new array domain, including "guards".
    
    Interval<1> N(1, n);
    Interval<2> vertDomain(N, N);

    // Resize the arrays.
    
    a_m.initialize(vertDomain);
    b_m.initialize(vertDomain);
    c_m.initialize(vertDomain);
  }

  Array<2, double, ETag> a_m, b_m, c_m;
};

template<class Mesh, class ETag>
class AbcStorage<Field<Mesh, double, ETag>, false>
{
public:

  void initializeStorage(int &n, int np, int ng) 
  {
    // Get new array domain, including "guards".

    Interval<1> N(1, n);
    Interval<2> vertDomain(N, N);

    DomainLayout<2> layout(vertDomain);

    Mesh mesh(layout, Vector<2>(0.0), Vector<2>(1.0, 1.0));

    Centering<2> vert = canonicalCentering<2>(VertexType, Continuous, AllDim);

    // Initialize the fields.

    a_m.initialize(vert, layout, mesh);
    b_m.initialize(vert, layout, mesh);
    c_m.initialize(vert, layout, mesh);
  }

  Field<Mesh, double, ETag> a_m, b_m, c_m;
};

template<class ETag>
class AbcStorage<Array<2, double, ETag>, true>
{
public:

  void initializeStorage(int &n, int np, int ng) 
  {
    n = (n / np) * np;
    
    // Get new array domain.
    
    Interval<1> N(1, n);
    Interval<2> newDomain(N, N);

    // Create the block sizes.
    
    Loc<2> blocks(np, np);

    // Create the partitioner:
    // We set the internal guard layers to ng_m, and don't allocate external
    // guard layers.  The internal guard layers are unnecessary.
    
    UniformGridPartition<2> partition(blocks,
                                      GuardLayers<2>(ng),
                                      GuardLayers<2>(0));

    // Create the layout.
    
    UniformGridLayout<2> layout(newDomain, partition, LayoutTag_t());

    // Create the arrays.
    
    a_m.initialize(layout);
    b_m.initialize(layout);
    c_m.initialize(layout);
  }

  Array<2, double, ETag> a_m, b_m, c_m;
};

template<class Mesh, class ETag>
class AbcStorage<Field<Mesh, double, ETag>, true>
{
public:

  void initializeStorage(int &n, int np, int ng) 
  {
    n = (n / np) * np;
    
    // Get new array domain.
    
    Interval<1> N(1, n);
    Interval<2> newDomain(N, N);

    // Create the block sizes.
    
    Loc<2> blocks(np, np);

    // Create the partitioner:
    // We set the internal guard layers to ng_m, and don't allocate external
    // guard layers.  The internal guard layers are unnecessary.
    UniformGridPartition<2> partition(blocks,
                                      GuardLayers<2>(ng),
                                      GuardLayers<2>(0));

    // Create the layout.
    
    UniformGridLayout<2> layout(newDomain, partition, LayoutTag_t());

    Mesh mesh(layout, Vector<2>(0.0), Vector<2>(1.0, 1.0));

    Centering<2> vert = canonicalCentering<2>(VertexType, Continuous, AllDim);

    // Initialize the fields.

    a_m.initialize(vert, layout, mesh);
    b_m.initialize(vert, layout, mesh);
    c_m.initialize(vert, layout, mesh);
  }

  Field<Mesh, double, ETag> a_m, b_m, c_m;
};

template<class Store>
struct StoreTraits
{
  typedef typename Store::Engine_t Engine_t;
  static const bool multiPatch = Engine_t::multiPatch;
};


//-----------------------------------------------------------------------------
// AbcBase class definitions.  We define this as a partially specialized
// class so it is easy to make subclasses work for Uniform MultiPatch
// (UMP) or not.
//-----------------------------------------------------------------------------

template<class Store>
class AbcBase :
  public Implementation,
  public AbcStorage<Store, StoreTraits<Store>::multiPatch>
{
public:

  // Dummy constructor.
  
  AbcBase(int np = 1, int ng = 0) 
  : np_m(np), ng_m(ng)
  { }
  
  // Initialize function gets the size and adjusts the arrays.
    
  void initialize(int n) 
  {
    // Save the problem size.
    
    n_m = n;

    // This call can decrease n_m to an integral multiple of np_m.
    this->initializeStorage(n_m, np_m, ng_m);

    // Set up domains for the internal cells.
    
    I = Interval<1>(1,n_m);
    J = Interval<1>(1,n_m);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 40 * double(n_m) * double(n_m) ); }

protected:

  // Problem check value.

  double check_m;

  // Problem size/number of patches.

  int n_m, np_m;
  
  // Internal guard layers.

  int ng_m;

  // Domains for stencil.
  
  Interval<1> I, J;
};


//-----------------------------------------------------------------------------
// AbcCppTran class definition, executing a C++ version of ForTran code.
//-----------------------------------------------------------------------------

template<class Store>
class AbcCppTran :
  public AbcBase<Store>
{
public:

  using AbcBase<Store>::a_m;
  using AbcBase<Store>::b_m;
  using AbcBase<Store>::c_m;

  // Constructor allows us to specify the number of patches for each direction.
  
  AbcCppTran(int np = 1)
    : AbcBase<Store>(np)
  { }

  // This is a C++Tran benchmark, perhaps using UMP.

  const char* type() const { return AbcBase<Store>::CppTranType(); }
  const char* qualification() const
  {
    typedef typename Store::Engine_t Engine_t;
    return ::qualification(a_m).c_str();
  }

  void run() 
  {
    // Run setup.

    runSetup();
    
    // Run kernel.

    for (int iter = 0; iter < 10; ++iter)
      {
	const double k = 1.0 / sqrt(static_cast<double>(iter+1));
	for (int j = 1; j <= this->n_m; ++j)
	  {
	    for (int i = 1; i <= this->n_m; ++i)
	      {
		a_m(i,j) = b_m(i,j) + k * c_m(i,j);
		c_m(i,j) = 0.5 * (a_m(i,j) - b_m(i,j));
	      }
	  }
      }
    
    // Save result for checking.
    
    this->check_m = a_m(this->n_m / 2, this->n_m / 2);
  }

  void runSetup()
  {
    for (int j = 1; j <= this->n_m; ++j)
      {
	for (int i = 1; i <= this->n_m; ++i)
	  {
	    b_m(i,j) = 0.0;
	    c_m(i,j) = 1000.0;
	  }
      }
  }
};


//-----------------------------------------------------------------------------
// AbcP2 class definition, executing Pooma data-parallel code.
//-----------------------------------------------------------------------------

template<class Store>
class AbcP2
  : public AbcBase<Store>
{
public:

  // Constructor allows us to specify the number of patches for each direction.
  
  AbcP2(int np = 1, int ng = 0)
    : AbcBase<Store>(np,ng),
      guarded_m(ng > 0)
  { }

  // This is a P2 benchmark, perhaps using UMP.

  const char* type() const { return AbcBase<Store>::P2Type(); }
  const char* qualification() const
  {
    typedef typename Store::Engine_t Engine_t;

    std::string qual = ::qualification(this->a_m);

    if (guarded_m)
    {
      PAssert(Engine_t::multiPatch);
      return ("GC" + qual).c_str();
    }
    else
    {
      return qual.c_str();
    }
  }

  void run() 
  {
    int k;

    // Run setup.

    runSetup();

    // Run kernel.
    
    for (int iter = 0; iter < 10; ++iter)
      {
	const double k = 1.0 / sqrt(static_cast<double>(iter+1));
	this->a_m = this->b_m + k * this->c_m;
	this->c_m = 0.5 * (this->a_m - this->b_m);
      }

    Pooma::blockAndEvaluate();

    // Save result for checking.
    
    this->check_m = a_m(this->n_m / 2, this->n_m / 2);
  }

  void runSetup()
  {
    // Run setup.
    
    this->b_m = 0.0;
    this->c_m = 1000.0;
    Pooma::blockAndEvaluate();
  }

private:
  bool guarded_m;
};

#endif // POOMA_BENCHMARKS_ABC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ABC.h,v $   $Author: richard $
// $Revision: 1.35 $   $Date: 2004/11/01 18:15:02 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
