# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

import types
import pyparsing

class Formula:
    """This class holds a formula attached to an object

    The main purpose of this class is to parse the formula and help
    with its recursive evaluation.

    """
    digits = '0123456789'
    lowercase = 'abcdefghijklmnopqrstuvwxyz'
    uppercase = lowercase.upper()
    word = pyparsing.Word(lowercase+uppercase, lowercase+uppercase+digits)
    extended_word = pyparsing.Word(lowercase+uppercase+digits)
    point = pyparsing.Literal('.')
    plus  = pyparsing.Literal("+")
    minus = pyparsing.Literal("-")
    mult  = pyparsing.Literal("*")
    div   = pyparsing.Literal("/")
    expon = pyparsing.Literal('^')
    lpar = pyparsing.Literal('(')
    rpar = pyparsing.Literal(')')
    sep = pyparsing.Literal(',')

    parsing_digits = pyparsing.Word(digits)
    number = (pyparsing.Group(parsing_digits + point + parsing_digits) |
              parsing_digits)

    name = pyparsing.Forward()
    name << (pyparsing.Group(word + pyparsing.OneOrMore(point+extended_word))
             | word)

    atom = name | number
    expr = pyparsing.Forward()
    fexpr = pyparsing.Forward()
    fatom = atom ^ fexpr

    expon_expr = pyparsing.Forward()
    expon_expr << (pyparsing.Group(fatom + expon + expon_expr) | fatom)

    mult_expr = pyparsing.Forward()
    mult_expr << (pyparsing.Group(expon_expr + (mult | div) + mult_expr)
                  | expon_expr)

    add_expr = pyparsing.Forward()
    add_expr << (pyparsing.Group(mult_expr + (plus | minus) + expr) | mult_expr)

    fexpr << pyparsing.Group(name + lpar +
                             (expr + (sep + expr)*(0,None))*(0,1) + rpar)
    expr << (add_expr ^ fexpr)

    # Other variables to be set:
    # isname
    # name_comps (if expression is a name or function)
    # all_names (tuple of name comps)
    # isnumber
    # isatom (if number or name)
    # operator
    def __init__(self, s):
        """Initialize with formula string"""
        self.s = s
        assert s is None or type(s) is types.StringType, s
        if s and s.strip(): self.isempty = False
        else:
            self.mark_empty()
            return
        # See if it is a functional expression
        try:
            self.parse_list = self.fexpr.parseString(s,
                                                     parseAll = True).asList()
            assert len(self.parse_list) == 1
            self.process_func(self.parse_list[0])
            return
        except pyparsing.ParseException: pass

        # Now see if it is an additive expression
        self.parse_list = self.add_expr.parseString(s,
                                             parseAll = True).asList()
        assert len(self.parse_list) == 1, self.parse_list
        if type(self.parse_list[0]) is types.StringType:
            self.process_atom(self.parse_list)
        else:
            sublist = self.parse_list[0]
            assert len(sublist) >= 3, sublist
            if sublist[1] == '.': self.process_atom(sublist)
            else: self.process_expr(sublist)

    def process_atom(self, comps):
        """Set variables given formula is atomic"""
        self.isatom = True
        self.operator = None
        self.subf = ()
        self.isfunc = False

        if comps[0][0] in self.digits:
            self.isnumber = True
            self.isname = False
            self.name_comps = None
            self.all_names = ()
        else:
            self.isnumber = False
            self.isname = True
            self.name_comps = tuple([elem for elem in comps if elem != '.'])
            self.all_names = (self.name_comps,)

    def process_expr(self, comps):
        """Process a complex expression"""
        assert len(comps) == 3, (self.s, comps)
        self.isnumber = False
        self.isname = False
        self.isatom = False
        self.isfunc = False
        self.operator = comps[1]
        self.argc = 2
        self.name_comps = None
        self.subf = (Formula(self.recreate_string(comps[0])),
                     Formula(self.recreate_string(comps[2])))
        self.set_all_names()

    def process_func(self, comps):
        """Process a function"""
        assert comps[1] == '(' and comps[-1] == ')', comps
        self.isatom = False
        self.isfunc = True

        inner_comps = [elem for elem in comps[2:-1] if elem != ',']
        self.argc = len(inner_comps)
        self.subf = [Formula(self.recreate_string(comp))
                     for comp in inner_comps]

        if type(comps[0]) is types.StringType:
            self.operator = (comps[0],)
        else:
            self.operator = tuple([elem for elem in comps[0] if elem != '.'])
        self.name_comps = self.operator
        self.set_all_names()

    def set_all_names(self):
        """Set self.all_names based on self.name_comps and subformulas"""
        if self.name_comps: l = [self.name_comps]
        else: l = []

        for subformula in self.subf: l.extend(subformula.all_names)
        self.all_names = tuple(l)

    def operator(self):
        """Return the main operator for the formula"""
        assert 0

    def argc(self):
        """Return the number of main terms/arguments in the formula"""
        assert 0

    def isname(self):
        """True if the formula is just a name"""
        assert 0

    def __str__(self):
        if self.s: return self.s
        return ""

    def recreate_string(self, nested_list):
        """Rebuild a subformula based on nested list of strings l"""
        def helper(l):
            assert l # should never be dealing with empty strings/lists
            if type(l) is types.StringType: return l
            else: return "".join([helper(elem) for elem in l])
        return helper(nested_list)
            
    def eval(self, binder):
        """Return the value of the formula

        binder is a function from name components to the values
        assigned to those names.
        """
        if self.isempty: return None
        if self.isatom:
            if self.isname: return binder(self.name_comps)
            assert self.isnumber
            return float(str(self))

        vals = [subf.eval(binder) for subf in self.subf]
        if self.isfunc:
            return binder(self.operator)(*vals)

        val1, val2 = vals
        if self.operator == '^': return val1**val2
        elif self.operator == '+': return val1+val2
        elif self.operator == '-': return val1-val2
        elif self.operator == '*': return val1*val2
        elif self.operator == '/': return val1/val2
        assert False, "Unknown operator "+self.operator

    def mark_empty(self):
        """Mark the formula as having no value (e.g., it's the empty string)"""
        self.isempty = True
        self.isatom = True
        self.operator = None
        self.subf = ()
        self.isfunc = False

        self.isnumber = False
        self.isname = False
        self.name_comps = None
        self.all_names = ()
