/********************************************************************************
                                   QueryToolView.cpp
                               -------------------
           begin                : lun jui 14 19:50:56 CEST 2003
       copyright                : (C) 2003 by Thomas Silvi
           email                : thomas.silvi@laposte.net
*********************************************************************************/

/********************************************************************************
*                                                                               *
*   This file is part of the Escort Tools Suite program.                        *
*                                                                               *
*   Escort Tools Suite is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by        *
*   the Free Software Foundation; either version 2 of the License, or           *
*   (at your option) any later version.                                         *
*                                                                               *
*   Escort Tools Suite is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
*   GNU General Public License for more details.                                *
*                                                                               *
*   You should have received a copy of the GNU General Public License           *
*   along with Escort Tools Suite; if not, write to the Free Software           *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   *
*                                                                               *
*********************************************************************************/

#include "QueryToolView.h"
#include "ToolSuiteConfig.h"
#include "QueryTool.h"
#include "OutputViewText.h"

#include <qvbox.h>
#include <qfile.h>

#include <kapplication.h>
#include <klocale.h>
#include <klineedit.h>
#include <kaction.h>
#include <kmessagebox.h>

#include <unistd.h>

#define PAGE_INPUT  0
#define PAGE_OUTPUT 1
#define PAGE_NOTICES 2


QueryToolView::QueryToolView(QueryToolDoc* pDoc, QWidget* pParent, 
	const char *pName, int pWflags) : QVBox(pParent, pName, pWflags)
{
	app = 0;
	doc = pDoc;
	batchIsBeingExecuted = false;
	batchExecutor = 0;
	creationDate = QDateTime::currentDateTime();

	ToolSuiteConfig *configTmp;
	configTmp = (doc->getApp())->getConfig();
	QString defaulWorkingDirectoryTmp = configTmp->getString("DefaultWorkingDirectory");
	
	directoryOpen = defaulWorkingDirectoryTmp;
	
	//	directorySaveInput = QString::null;
	directorySaveInput = defaulWorkingDirectoryTmp;
	fileNameSaveInput = QString::null;
	
	//	directorySaveOutput = QString::null;
	directorySaveOutput = defaulWorkingDirectoryTmp;
	fileNameSaveOutput = QString::null;

	// query input & output
	
	tw_queryIO = new QTabWidget (this);
	
	queryInput = new TextEditQuery();

	queryOutput = new OutputViewText(doc->getName());

	noticeMessages = new KTextEdit();
	
	tw_queryIO->addTab(queryInput,i18n("Input"));
	tw_queryIO->addTab(queryOutput,i18n("Output"));
	tw_queryIO->addTab(noticeMessages, i18n("Notices"));
	
	queryInput->init();
	initOutput();
	initNoticeMessages();
	
	// statusbar
	
	statusBarMesg = new KLineEdit(this);
	statusBarMesg->setMaxLength(64);
	statusBarMesg->setReadOnly(true);
//    statusBarMesg->setText(pDoc->getName());

	// set notice processor

	PGconn *connTmp;
	connTmp = (pDoc->getEngine())->getPGConnection();
	PQsetNoticeProcessor(connTmp,globalNoticeProcessor,this);
	connTmp = 0;
	
	// misc

	this->setCaption(pDoc->getName());
	
	connect(tw_queryIO, SIGNAL(currentChanged(QWidget *)),
		this, SLOT(switchView()));
	
	switchToInput();	
}

QueryToolView::~QueryToolView()
{
	app = 0;
	
	if (doc != 0)
	{
		delete doc;
		doc = 0;
	}
	
	if (batchExecutor !=0)
	{
		delete batchExecutor;
		batchExecutor = 0;
	}
}

QueryToolDoc * QueryToolView::getDocument() const
{
	return doc;
}

void QueryToolView::setDocument(QueryToolDoc *pDoc)
{
	doc = pDoc;
	return;
}

void QueryToolView::openInput(QString &pDirectory, QString &pFileName)
{
	QString pathfile;
	
	// when we open a file, we reset the saveFileName
	// in order to avoid to loose the previous script file
	
	ToolSuiteConfig *configTmp;
	configTmp = (doc->getApp())->getConfig();
	QString defaulWorkingDirectoryTmp = configTmp->getString("DefaultWorkingDirectory");
	
	
	directorySaveInput = defaulWorkingDirectoryTmp;
	fileNameSaveInput = QString::null;
	
	directorySaveOutput = defaulWorkingDirectoryTmp;
	fileNameSaveOutput = QString::null;
	
	// continue with file loading
	
	setStatusBarMsg("");
	
	pathfile = pDirectory + "/" + pFileName;
	
#ifdef DEBUG_APP
	printf ("DEBUG_APP QueryToolView::openInput >Loading >%s<\n",pathfile.ascii());
#endif
	
	QFile file(pathfile);
	
	if (!file.open(IO_ReadOnly))
	{
		// error
		KMessageBox::error (this,
			i18n("Error : Can not open file <%1> !").arg(pFileName));
		return;
	}
	
	QTextStream stream(&file);
	
	
	queryInput->setText((stream.read()).ascii());
	
	queryInput->selectAll(true);
	queryInput->setCurrentFont(QFont("Courier",10,QFont::Normal));
	queryInput->setTextFormat(PlainText);
	queryInput->setWordWrap(QTextEdit::NoWrap);
	
	file.close();
	
	queryInput->selectAll(false);
	queryInput->removeLastCharacter();
	
	directoryOpen = pDirectory;
	fileNameOpenInput = pFileName;
	
	setStatusBarMsg(i18n("File <%1> loaded").arg(pFileName));
	
	return;
}

void QueryToolView::saveInput(QString &pDirectory, QString &pFileName)
{
	QString pathfile;
	
	setStatusBarMsg("");
	
	// if pFileName does not end with .sql, add it

	if (pFileName.endsWith(".sql") == false)
	{
		pFileName = pFileName + ".sql";
	}



	pathfile = pDirectory + "/" + pFileName;
	
#ifdef DEBUG_APP
	printf ("DEBUG_APP QueryToolView::saveInput> Save Input >%s<\n",pathfile.ascii());
#endif
	
	QFile file(pathfile);
	
	if (!file.open(IO_WriteOnly))
	{
		// error
		KMessageBox::error (this,
			i18n("Error : Can not save input file <%1> !").arg(pFileName));
		return;
	}
	
	QTextStream stream(&file);
	stream << queryInput->text();
	file.close();
	
	directorySaveInput = pDirectory;
	fileNameSaveInput = pFileName;
	
	setStatusBarMsg(i18n("File <%1> saved").arg(fileNameSaveInput));
	
	return;
}

void QueryToolView::saveOutput(QString &pDirectory, QString &pFileName)
{
	QString pathfile;
	
	setStatusBarMsg("");
	
	pathfile = pDirectory + "/" + pFileName;

#ifdef DEBUG_APP
	printf ("DEBUG_APP QueryToolView::saveOutput> Save Output >%s<\n",pathfile.ascii());
#endif
	
	QFile file(pathfile);
	
	if (!file.open(IO_WriteOnly))
	{
		// error
		KMessageBox::error (this,
			i18n("Error : Can not save output file <%1> !").arg(pFileName));
		return;
	}
	
	queryOutput->saveOutput(file);

	file.close();
	
	directorySaveOutput = pDirectory;
	fileNameSaveOutput = pFileName;
	
	setStatusBarMsg(i18n("File <%1> saved").arg(fileNameSaveOutput));
	
	return;
}

QString QueryToolView::getDirectoryOpen()
{
	return directoryOpen;
}

QString QueryToolView::getFileNameOpenInput()
{
	return fileNameOpenInput;
}

QString QueryToolView::getDirectorySaveInput()
{
	return directorySaveInput;
}

QString QueryToolView::getFileNameSaveInput()
{
	return fileNameSaveInput;
}

QString QueryToolView::getDirectorySaveOutput()
{
	return directorySaveOutput;
}

QString QueryToolView::getFileNameSaveOutput()
{
	return fileNameSaveOutput;
}

signed char QueryToolView::getActiveTab()
{
	signed char result;
	
	result = -1;

	switch (tw_queryIO->currentPageIndex())
	{
		case PAGE_INPUT :
		{
			result = PAGE_INPUT;
			break;
		}
		case PAGE_OUTPUT :
		{
			result = PAGE_OUTPUT;
			break;
		}
		case PAGE_NOTICES :
		{
			result = PAGE_NOTICES;
			break;
		}
		default :
		{
			result = -1;
			break;
		}
	}

	return result;
}

void QueryToolView::setApp(KApplication *pApp)
{
	app = pApp;

	if (queryOutput !=0)
	{
		queryOutput->setApplication(app);
	}

	return;
}

bool QueryToolView::isBatchRunning()
{
	return batchIsBeingExecuted;
}

void QueryToolView::executeInput()
{
	Batch *batchTmp;
	QString batchQuery, errorMessage;

	if (queryOutput->specialInit(errorMessage) == false) 
	{
	 	KMessageBox::error (this, errorMessage);
		return;
	}
	
	// init
	
	if (batchIsBeingExecuted == true)
	{
		KMessageBox::error (this, i18n("Error : a batch is already running !"));
		return;
	}
	
	batchIsBeingExecuted = true;
	
	if (batchExecutor != 0)
	{
		delete batchExecutor;
	}
	
	
	resetOutput();
	resetNoticeMessages();
	
	// Get Batch
	
	if (queryInput->hasSelectedText())
	{
		batchQuery = queryInput->selectedText();
	}
	else
	{
		batchQuery = queryInput->text();
	}
	
	batchTmp = new Batch();
	if (batchTmp == 0)
	{
#ifdef DEBUG_APP
	printf("DEBUG_APP QueryToolView::executeInput> Batch creation failed");
#endif
			return;
	}
	
	batchTmp->build(batchQuery);
	
	// init & run batchExecutor
	
	batchExecutor = new BatchExecutorThread();
	
	batchExecutor->setBatch(batchTmp);
	batchExecutor->setView(this);
	batchExecutor->setEngine(doc->getEngine());
	batchExecutor->setConfig((doc->getApp())->getConfig());
	
	setStatusBarMsg(i18n("Running"));
	batchExecutor->start();
	
	return;
}

void QueryToolView::notice(QString &pMessage)
{
	QWidget *pageTmp;
	
	pageTmp = tw_queryIO->page(PAGE_NOTICES);
	
	app->lock();
	noticeMessages->append(pMessage);
	tw_queryIO->setTabLabel(pageTmp,i18n("Notices") + "*");
	app->unlock();
#ifdef DEBUG_APP
	printf ("DEBUG_APP QueryToolView::notice> Notice : %s\n",pMessage.ascii());
#endif
	return;
}

void QueryToolView::initOutput()
{
	queryOutput->initOutput();
	resetOutput();
	return;
}

void QueryToolView::initNoticeMessages()
{
	noticeMessages->setTextFormat(PlainText);
	noticeMessages->setCurrentFont(QFont("Courier",10));
	noticeMessages->setWordWrap(QTextEdit::NoWrap);
	noticeMessages->setReadOnly(TRUE);
	noticeMessages->setPaletteBackgroundColor(QColor("white"));
	resetNoticeMessages();
	return;
}

void QueryToolView::switchToInput()
{
	app->lock();
	tw_queryIO->setCurrentPage(PAGE_INPUT);
	app->unlock();
	return;
}

void QueryToolView::switchToOutput()
{
	app->lock();
	tw_queryIO->setCurrentPage(PAGE_OUTPUT);
	app->unlock();
	return;
}

void QueryToolView::switchToNoticeMessages()
{
	app->lock();
	tw_queryIO->setCurrentPage(PAGE_NOTICES);
	app->unlock();
	return;
}

void QueryToolView::switchView()
{
	if (tw_queryIO->currentPageIndex() == PAGE_INPUT)
	{
		queryInput->setFocus();
		tw_queryIO->repaint();
	}
	
	return;
}

void QueryToolView::stopBatch()
{
	batchExecutor->cancelBatchExecution();
	statusBarMesg->setText(i18n("Batch stopped"));
	batchIsBeingExecuted = false;
	tw_queryIO->repaint();
	return;
}

void QueryToolView::setStatusBarMsg(QString pMsg)
{
	if (statusBarMesg == 0)
	{
		return;
	}
	
	statusBarMesg->setText(pMsg);
	return;
}

void QueryToolView::showResuts(PGresult *pResults)
{
	ToolSuiteConfig *configTmp;
	unsigned int nRows, nrowsConfig;
//      int result;
	QString sTmp;

	configTmp = (doc->getApp())->getConfig();

	if (configTmp->getBool("PromptForShowingResultsWithMoreThanNRows") == true)
	{
		nrowsConfig = configTmp->getUInt("NRows");
		nRows = PQntuples(pResults);
//printf ("Rows : %d MaxRows : %d\n", nRows, nrowsConfig);

		if (nRows > nrowsConfig)
		{
			queryOutput->printOutput(
				i18n("Resultset with %1 rows not showed.").arg(nRows));
// resultsets with n rows not showed.
			return;
/*
			sTmp.setNum(nRows);
			result = KMessageBox::warningYesNo (this, i18n("The results contains \
				%1 rows. Do you want to see it ? \
				").arg(sTmp));
			if (result == KMessageBox::No)
			{
				// nothing is shown and byebye showResults
				return;
			}
*/
		}
	}

	queryOutput->showResuts(pResults);
	PQclear(pResults);

	tw_queryIO->repaint();

	return;
}

void QueryToolView::printOutput(QString pStr)
{
	app->lock();
	queryOutput->printOutput(pStr);
	app->unlock();
	return;
}

void QueryToolView::batchExecuted()
{
	Batch *batchTmp;

	batchIsBeingExecuted = false;
	batchTmp = batchExecutor->getBatch();

	if (batchTmp == 0)
	{
		setStatusBarMsg(i18n("Batch executed"));
	}
	else 
	{
		setStatusBarMsg(i18n("Batch executed with %1 error(s)").arg(batchTmp->errorsCount));
	}
	tw_queryIO->repaint();

	return;
}

void QueryToolView::resetOutput()
{
	queryOutput->resetOutput();
	return;
}

void QueryToolView::resetNoticeMessages()
{
	ToolSuiteConfig *configTmp;
	QWidget *pageTmp;
	
	configTmp = (doc->getApp())->getConfig();
	
	if (configTmp->getBool("ClearNoticesOutput") == true)
	{
		noticeMessages->clear();
	}
	
	pageTmp = tw_queryIO->page(PAGE_NOTICES);
	tw_queryIO->setTabLabel(pageTmp,i18n("Notices"));
	
	return;
}

QDateTime QueryToolView::getCreationDate() const
{
	return creationDate;
}

QueryToolViewConfig * QueryToolView::getQueryToolViewConfig() const
{
	QueryToolViewConfig * result;

	result = new QueryToolViewConfig();

	result->size = this->size();

	return result;
}

void QueryToolView::applyQueryToolViewConfig (QueryToolViewConfig *pViewConfig)
{
	if (pViewConfig == 0)
	{
		return;
	}

	this->resize(pViewConfig->size);

	return;
}

bool QueryToolView::askIfDisconnect()
{
	int tmpResult;
	bool result;

	tmpResult = KMessageBox::warningYesNo(this, i18n("Do you want to close the connection ?"),
			i18n("Warning : disconnecting"));

	if (tmpResult == KMessageBox::No)
	{
		result = false;
	}
	else
	{
		result = true;
	}

	return result;
}

void QueryToolView::closeEvent(QCloseEvent *pEvent)
{
	if (batchIsBeingExecuted == true)
	{
		// we have to stop the batch
		// if not, the thread will send data to an unexistant object 
		pEvent->ignore();
		this->setStatusBarMsg(i18n("Error : a batch is running."));
	}
	else
	{
		if (this->askIfDisconnect() == true)
		{
			pEvent->accept();
		}
		else
		{
			pEvent->ignore();
		}
	}

	return;
}





void globalNoticeProcessor(void *arg, const char *message)
{
	// the message parameter contains the address
	// of the view.
	QueryToolView *viewTmp;
	QString sTmp(message);
	
	viewTmp = (QueryToolView*) arg;
	viewTmp->notice(sTmp);
	
	return;
}





	
