/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope tobject it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "object.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include <librsvg/rsvg.h>
#include <stdarg.h>
DoodleHashObject* doodlehash_object_new()
{
  DoodleHashObject *o = g_malloc (sizeof (DoodleHashObject));
  if (!o)
    return NULL;
  memset (o, 0, sizeof (DoodleHashObject));
  return o;
}

DoodleHashObject *doodlehash_object_copy(const DoodleHashObject *orig)
{
  DoodleHashObject *object = doodlehash_object_new();
  object->width = orig->width;
  object->height = orig->height;
  object->data = g_strdup(orig->data);
  object->data_len = orig->data_len;
  object->primary_stroke_colour = g_strdup(orig->primary_stroke_colour);
  object->primary_fill_colour = g_strdup(orig->primary_fill_colour);
  object->secondary_stroke_colour = g_strdup(orig->secondary_stroke_colour);
  object->secondary_fill_colour = g_strdup(orig->secondary_fill_colour);
  object->primary_stroke_colours = g_strdupv(orig->primary_stroke_colours);
  object->primary_fill_colours = g_strdupv(orig->primary_fill_colours);
  object->secondary_stroke_colours = g_strdupv(orig->secondary_stroke_colours);
  object->secondary_fill_colours = g_strdupv(orig->secondary_fill_colours);
  object->num_colours = orig->num_colours;
  return object;
}


int doodlehash_object_load_from_svg_file(DoodleHashObject *o, const gchar *file)
{
  if (g_file_get_contents(file, &o->data, &o->data_len, NULL) == FALSE)
    return -1;

  if (doodlehash_object_set_width_and_height_from_svg_file(o, file) != 0)
    return -2;
  return get_colours((const guint8*)o->data, o->data_len, 
                     &o->primary_stroke_colours, &o->primary_fill_colours, 
                     &o->secondary_stroke_colours, &o->secondary_fill_colours, 
                     &o->num_colours);
}

void doodlehash_object_get_width_and_height(DoodleHashObject* object, guint32* width, guint32* height)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  *width = o->width;
  *height = o->height;
}

guint32 doodlehash_object_get_width(DoodleHashObject* object)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  return o->width;
}

guint32 doodlehash_object_get_height(DoodleHashObject* object)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  return o->height;
}

void doodlehash_object_set_width(DoodleHashObject* object, guint32 width)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  o->width = width;
}

void doodlehash_object_set_height(DoodleHashObject* object, guint32 height)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  o->height = height;
}

void doodlehash_object_set_random_colours(DoodleHashObject* object, guint code)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  if (o->num_colours > 0)
    {
      guint32 i = rand() % o->num_colours;
      if (code == DOODLEHASH_COLOUR_PRIMARY)
        doodlehash_object_set_colours(object, code, o->primary_stroke_colours[i], o->primary_fill_colours[i]);
      else if (code == DOODLEHASH_COLOUR_SECONDARY)
        doodlehash_object_set_colours(object, code, o->secondary_stroke_colours[i], o->secondary_fill_colours[i]);
    }
}

void doodlehash_object_set_colours(DoodleHashObject* object, guint code, char *s, char *f)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  char *newdata = NULL;
  size_t newdatalen = 0;
  if (code == DOODLEHASH_COLOUR_PRIMARY)
    {
      colourify_svg_file(o->data, o->data_len, &newdata, &newdatalen, 
                         o->primary_stroke_colour, o->primary_fill_colour, 
                         s, f, code);
      if (o->primary_stroke_colour)
        g_free (o->primary_stroke_colour);
      o->primary_stroke_colour = g_strdup (s);
      if (o->primary_fill_colour)
        g_free (o->primary_fill_colour);
      o->primary_fill_colour = g_strdup (f);
    }
  else if (code == DOODLEHASH_COLOUR_SECONDARY)
    {
      colourify_svg_file(o->data, o->data_len, &newdata, &newdatalen, 
                         o->secondary_stroke_colour, o->secondary_fill_colour,
                         s, f, code);
      if (o->secondary_stroke_colour)
        g_free (o->secondary_stroke_colour);
      o->secondary_stroke_colour = g_strdup (s);
      if (o->secondary_fill_colour)
        g_free (o->secondary_fill_colour);
      o->secondary_fill_colour = g_strdup (f);
    }
  g_free (o->data);
  o->data = newdata;
  o->data_len = newdatalen;
}

gchar *doodlehash_object_get_fill_colour(DoodleHashObject* object, guint code)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  if (code == DOODLEHASH_COLOUR_PRIMARY)
    return o->primary_fill_colour;
  else if (code == DOODLEHASH_COLOUR_SECONDARY)
    return o->secondary_fill_colour;
  return NULL;
}

gchar *doodlehash_object_get_stroke_colour(DoodleHashObject* object, guint code)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  if (code == DOODLEHASH_COLOUR_PRIMARY)
    return o->primary_stroke_colour;
  else if (code == DOODLEHASH_COLOUR_SECONDARY)
    return o->secondary_stroke_colour;
  return NULL;
}

void doodlehash_object_free(DoodleHashObject* object)
{
  DoodleHashObject *o = (DoodleHashObject*) object;
  if (o && o->data)
    g_free (o->data);
  if (o && o->primary_stroke_colour)
      g_free (o->primary_stroke_colour);
  if (o && o->primary_fill_colour)
      g_free (o->primary_fill_colour);
  if (o && o->secondary_stroke_colour)
      g_free (o->secondary_stroke_colour);
  if (o && o->secondary_fill_colour)
      g_free (o->secondary_fill_colour);

  if (o && o->primary_stroke_colours)
    g_strfreev(o->primary_stroke_colours);
  if (o && o->primary_fill_colours)
    g_strfreev(o->primary_fill_colours);
  if (o && o->secondary_stroke_colours)
    g_strfreev(o->secondary_stroke_colours);
  if (o && o->secondary_fill_colours)
    g_strfreev(o->secondary_fill_colours);
  g_free (object);
}

void get_matching_colours(gchar **in_ps, gchar **in_pf, gchar **in_ss, gchar **in_sf, DoodleHashObject *o, gchar ***out_ps, gchar ***out_pf, gchar ***out_ss, gchar ***out_sf)
{
  if (o->num_colours == 0)
    {
      *out_ps = g_strdupv(in_ps);
      *out_pf = g_strdupv(in_pf);
      *out_ss = g_strdupv(in_ss);
      *out_sf = g_strdupv(in_sf);
      return;
    }
  guint count = g_strv_length(in_ps);
  guint ob_count = o->num_colours;
  int i, j, num = 0;
  for (i = 0; i < count; i++)
    for (j = 0; j < ob_count; j++)
      {
        if (strcmp(in_ps[i],o->primary_stroke_colours[j])== 0 &&
            strcmp(in_pf[i],o->primary_fill_colours[j])== 0 &&
            strcmp(in_ss[i],o->secondary_stroke_colours[j]) == 0 &&
            strcmp(in_sf[i],o->secondary_fill_colours[j]) == 0) 
          {
            num++;
            *out_ps = g_realloc (*out_ps, num * sizeof (gchar*));
            (*out_ps)[num-1] = g_strdup(in_ps[i]);
            *out_pf = g_realloc (*out_pf, num * sizeof (gchar*));
            (*out_pf)[num-1] = g_strdup(in_pf[i]);
            *out_ss = g_realloc (*out_ss, num * sizeof (gchar*));
            (*out_ss)[num-1] = g_strdup(in_ss[i]);
            *out_sf = g_realloc (*out_sf, num * sizeof (gchar*));
            (*out_sf)[num-1] = g_strdup(in_sf[i]);
          }
      }
  if (num > 0)
    {
      num++;
      *out_ps = g_realloc (*out_ps, num * sizeof (gchar*));
      (*out_ps)[num-1] = NULL;
      *out_pf = g_realloc (*out_pf, num * sizeof (gchar*));
      (*out_pf)[num-1] = NULL;
      *out_ss = g_realloc (*out_ss, num * sizeof (gchar*));
      (*out_ss)[num-1] = NULL;
      *out_sf = g_realloc (*out_sf, num * sizeof (gchar*));
      (*out_sf)[num-1] = NULL;
    }
}

void doodlehash_object_set_random_matching_colours(DoodleHashObject *first, ...)
{
  va_list args;
  va_start (args, first);
  DoodleHashObject *o;
  DoodleHashObject *next;
  next = va_arg(args, DoodleHashObject*);
  gchar **in_ps = g_strdupv(first->primary_stroke_colours);
  gchar **in_pf = g_strdupv(first->primary_fill_colours);
  gchar **in_ss = g_strdupv(first->secondary_stroke_colours);
  gchar **in_sf = g_strdupv(first->secondary_fill_colours);
  for (o = next; o != NULL; o = next)
    {
      next = va_arg(args, DoodleHashObject*);
      gchar **out_ps = NULL,  **out_pf = NULL, **out_ss = NULL, **out_sf = NULL;
      get_matching_colours(in_ps, in_pf, in_ss, in_sf, o,
                           &out_ps, &out_pf, &out_ss, &out_sf);
      g_strfreev(in_ps); g_strfreev(in_pf);
      g_strfreev(in_ss); g_strfreev(in_sf);
      in_ps = out_ps; in_pf = out_pf;
      in_ss = out_ss; in_sf = out_sf;
    }
  va_end (args);
  //now we have a list of random colours that happen to match across all objs
  guint count = g_strv_length(in_ps);
  if (count > 0)
    {
      guint num = rand() % count;
      char *ps = in_ps[num];
      char *pf = in_pf[num];
      char *ss = in_ss[num];
      char *sf = in_sf[num];
      va_start (args, first);
      for (o = first; o != NULL; o = next)
        {
          next = va_arg(args, DoodleHashObject*);
          if (doodlehash_object_has_data(o))
            {
              doodlehash_object_set_colours (o, DOODLEHASH_COLOUR_PRIMARY, 
                                             ps, pf);
              doodlehash_object_set_colours (o, DOODLEHASH_COLOUR_SECONDARY, 
                                             ss, sf);
            }
        }
      va_end (args);
    }
  g_strfreev(in_ps); g_strfreev(in_pf);
  g_strfreev(in_ss); g_strfreev(in_sf);
}

guint32 doodlehash_object_get_width_from_svg_file(const char *file)
{
  DoodleHashObject *ob = doodlehash_object_new();
  if (doodlehash_object_set_width_and_height_from_svg_file(ob, file) != 0)
    return 0;
  guint32 width = doodlehash_object_get_width(ob);
  doodlehash_object_free(ob);
  return width;
}

int doodlehash_object_set_width_and_height_from_svg_file(DoodleHashObject *o, const char *file)
{
  GError *e = NULL;
  if (g_file_get_contents(file, &o->data, &o->data_len, NULL) == FALSE)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data ((const guint8*)o->data, o->data_len, &e);
  if (!handle || e)
    return -2;
  RsvgDimensionData dim;
  rsvg_handle_get_dimensions (handle, &dim);
  o->height = dim.height;
  o->width = dim.width;
  rsvg_handle_close (handle, &e);
  return 0;
}
              
int doodlehash_object_has_data(DoodleHashObject *object)
{
  return (object->data_len > 0) ? 1 : 0;
}
void doodlehash_object_scale(DoodleHashObject *object, double scale)
{
  object->height *= scale;
  object->width *= scale;
}
