/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "earrings.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include <stdlib.h>
#include "util.h"
#include "object.h"

DoodleHashEarRingPair* doodlehash_earrings_new()
{
  DoodleHashEarRingPair *e = g_malloc (sizeof (DoodleHashEarRingPair));
  if (!e)
    return NULL;
  memset (e, 0, sizeof (DoodleHashEarRingPair));
  e->left = g_malloc (sizeof (DoodleHashEarRing));
  if (!e->left)
    {
      g_free (e);
      return NULL;
    }
  memset (e->left, 0, sizeof (DoodleHashEarRing));
  e->left->base = doodlehash_object_new();
  e->right = g_malloc (sizeof (DoodleHashEarRing));
  if (!e->right)
    {
      g_free (e->left);
      g_free (e);
      return NULL;
    }
  memset (e->right, 0, sizeof (DoodleHashEarRing));
  e->right->base = doodlehash_object_new();
  return e;
}
void doodlehash_earring_copy(const DoodleHashEarRing*orig, DoodleHashEarRing *copy)
{
  g_free (copy->base);
  copy->base = doodlehash_object_copy(orig->base);
  copy->connect_to_ear_x = orig->connect_to_ear_x;
  copy->connect_to_ear_y = orig->connect_to_ear_y;
}

DoodleHashEarRingPair* doodlehash_earrings_copy(const DoodleHashEarRingPair *orig)
{
  DoodleHashEarRingPair *earrings = doodlehash_earrings_new();
  if (orig->left)
    doodlehash_earring_copy(orig->left, earrings->left);
  if (orig->right)
    doodlehash_earring_copy(orig->right, earrings->right);
  return earrings;
}

int doodlehash_earring_load_from_svg(DoodleHashEarRing *e, const char *file)
{
  if (doodlehash_object_load_from_svg_file(e->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)e->base->data, 
                                                 e->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-ear", 
                         &e->connect_to_ear_x, &e->connect_to_ear_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashEarRingPair* doodlehash_earrings_load_from_svg_file(const gchar *left, const gchar *right)
{
  DoodleHashEarRingPair *earring = doodlehash_earrings_new();
  if (doodlehash_earring_load_from_svg(earring->left, left) == 0 &&
      doodlehash_earring_load_from_svg(earring->right, right) == 0)
    return earring;
  else
    {
      doodlehash_earrings_free(earring);
      return NULL;
    }
}

void doodlehash_earrings_get_width_and_height(DoodleHashEarRingPair* earring, guint side, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base, width, height);
}

void doodlehash_earrings_set_width(DoodleHashEarRingPair* er, guint side, guint32 width)
{
  DoodleHashEarRing *e = (side == DOODLEHASH_SIDE_LEFT) ? er->left : er->right;
  double scale = (double) width / (double)e->base->width;
  e->connect_to_ear_x *= scale;
  doodlehash_object_set_width (e->base, e->base->width * scale);
}

guint32 doodlehash_earrings_get_width(DoodleHashEarRingPair* earring, guint side)
{
  return doodlehash_object_get_width(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base);
}

guint32 doodlehash_earrings_get_height(DoodleHashEarRingPair* earring, guint side)
{
  return doodlehash_object_get_height(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base);
}

void doodlehash_earrings_set_height(DoodleHashEarRingPair* er, guint side, guint32 height)
{
  DoodleHashEarRing *e = (side == DOODLEHASH_SIDE_LEFT) ? er->left : er->right;
  double scale = (double) height / (double)e->base->height;
  e->connect_to_ear_x *= scale;
  doodlehash_object_set_height (e->base, e->base->height * scale);
}

void doodlehash_earrings_get_ear_connection_point(DoodleHashEarRingPair* earring, guint side, guint32* x, guint32* y)
{
  DoodleHashEarRing *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = earring->left;
  else
    e = earring->right;
  *x = e->connect_to_ear_x;
  *y = e->connect_to_ear_y;
}

void doodlehash_earrings_set_ear_connection_point(DoodleHashEarRingPair* earring, guint side, guint32 x, guint32 y)
{
  DoodleHashEarRing *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = earring->left;
  else
    e = earring->right;
  e->connect_to_ear_x = x;
  e->connect_to_ear_y = y;
}

guint32 doodlehash_earrings_get_ear_connection_point_x(DoodleHashEarRingPair* earring, guint side)
{
  return (side == DOODLEHASH_SIDE_LEFT) ?  earring->left->connect_to_ear_x : earring->right->connect_to_ear_x;
}

guint32 doodlehash_earrings_get_ear_connection_point_y(DoodleHashEarRingPair* earring, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? earring->left->connect_to_ear_y : earring->right->connect_to_ear_y;
}

void doodlehash_earrings_set_random_matching_colours(DoodleHashEarRingPair *earring)
{
  doodlehash_object_set_random_matching_colours(earring->left->base, 
                                                earring->right->base,
                                                NULL);
}
void doodlehash_earrings_set_random_colours(DoodleHashEarRingPair* earring, guint code)
{
  doodlehash_object_set_random_colours(earring->left->base, code);
  doodlehash_object_set_random_colours(earring->right->base, code);
}
void doodlehash_earrings_set_colours(DoodleHashEarRingPair* earring, guint side, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base, code, s, f);
}

gchar *doodlehash_earrings_get_fill_colour(DoodleHashEarRingPair* earring, guint side, guint code)
{
  return doodlehash_object_get_fill_colour(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base, code);
}

gchar *doodlehash_earrings_get_stroke_colour(DoodleHashEarRingPair* earring, guint side, guint code)
{
  return doodlehash_object_get_stroke_colour(side == DOODLEHASH_SIDE_LEFT ? earring->left->base : earring->right->base, code);
}

void doodlehash_earrings_scale(DoodleHashEarRingPair *er, guint side, double scale)
{
  DoodleHashEarRing *e = (side == DOODLEHASH_SIDE_LEFT) ? er->left : er->right;
  e->connect_to_ear_x *= scale;
  e->connect_to_ear_y *= scale;
  doodlehash_object_scale(e->base, scale);
}

void doodlehash_earrings_free(DoodleHashEarRingPair* e)
{
  if (e && e->left && e->left->base)
    doodlehash_object_free (e->left->base);
  if (e && e->right && e->right->base)
    doodlehash_object_free (e->right->base);
  if (e && e->right)
    g_free (e->right);
  if (e && e->left)
    g_free (e->left);
  g_free (e);
}

