/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "util.h"
#include <string.h>
#include <unistd.h>
#include <librsvg/rsvg.h>

xmlNodePtr findSVGNode(xmlNodePtr p, char * name)
{
  xmlNodePtr f = p;
  if(p == NULL || name == NULL) return NULL;

  for(p=p->children; p!= NULL; p=p->next)
    {
      if(p->name && strcmp((char*)p->name,name) == 0 )
          return p;
    }
  for(p=f->children; p!= NULL; p=p->next)
    {
      xmlNodePtr done = findSVGNode(p, name);
      if (done)
        return done;
    }
  return NULL;
}

int find_connect_point(RsvgHandle *handle, char *name, unsigned int *x, unsigned int *y)
{
  int retval = -1;
  RsvgPositionData pos;
  if (rsvg_handle_get_position_sub (handle, &pos, name))
    {
      *x = (unsigned int) pos.x;
      *y = (unsigned int) pos.y;
      retval = 0;
    }
  return retval;
}

gchar* get_svg_path(const char *subdir)
{
 return g_build_filename(getenv("DOODLEHASH_PATH") ? getenv("DOODLEHASH_PATH") : DOODLEHASH_DATADIR, subdir, NULL);
}
  
void colourify_svg(const char *indata, size_t indatalen, char **outdata, size_t *outdatalen, char *in_color1, char *out_color1, char *in_color2, char *out_color2)
{
  const gchar *contents = indata;
  gchar **parts = g_strsplit(contents, in_color1, -1);
  if (parts)
    {
      gchar *new_contents = g_strjoinv(out_color1, parts);
      g_strfreev(parts);
      parts = g_strsplit(new_contents, in_color2, -1);
      g_free (new_contents);
      if (parts)
        {
          *outdata = g_strjoinv(out_color2, parts);
          *outdatalen = strlen (*outdata);
          g_strfreev(parts);
        }
    }
}

void colourify_svg_file (const char *indata, size_t indatalen, char **outdata, size_t *outdatalen, char *prev_stroke, char *prev_fill, char *stroke, char *fill, int primary)
{
  if (stroke && fill)
    {
      gchar *ps = NULL;
      if (prev_stroke)
        ps = g_strdup_printf ("stroke:%s", prev_stroke);
      gchar *pf = NULL;
      if (prev_fill)
        pf = g_strdup_printf ("fill:%s", prev_fill);
      gchar *s = g_strdup_printf ("stroke:%s", stroke);
      gchar *f = g_strdup_printf ("fill:%s", fill);
      if (primary)
        colourify_svg(indata, indatalen, outdata, outdatalen,
                      ps ? ps: "stroke:#d00d1e", s,
                      pf ? pf: "fill:#114511", f);
      else
        colourify_svg(indata, indatalen, outdata, outdatalen, 
                      ps ? ps: "stroke:#f1dd1e", s, 
                      pf ? pf: "fill:#00d1e5", f);
      g_free (s);
      g_free (f);
      g_free (ps);
      g_free (pf);
    }
}

int is_valid_colour(char *colour)
{
  /* Looks like:#000000 #000000 #000000 #000000*/
  if (colour[0] == '#' && strlen (colour) == 31 && 
      colour[7] == ' ' && colour[8] == '#'  &&
      colour[15] == ' ' && colour[16] == '#' &&
      colour[23] == ' ' && colour[24] == '#' && 
      strspn (colour, "0123456789abcdefABCDEF# ") == 31)
    return 1;
  return 0;
}

int get_colours(const guint8* data, gsize data_len, gchar ***primary_stroke, gchar ***primary_fill, char ***secondary_stroke, gchar ***secondary_fill, guint32 *num_colours)
{
  GError *e = NULL;
  RsvgHandle *handle = rsvg_handle_new_from_data(data, data_len, &e);
  if (!handle || e)
    return -1;
  const gchar *desc = rsvg_handle_get_metadata(handle);
  if (desc)
    {
      int i = 0;
      gchar **lines = g_strsplit(desc, "\n", -1);
      gchar **ptr = lines;
      while (*ptr != NULL)
        {
          char *line = strchr(*ptr, '#');
          if (line)
            {
              g_strstrip(line);
              char *tmp;
              if ((tmp=strchr (line, '<')))
                *tmp = 0;

              if (is_valid_colour(line))
                {
                  char ps[16];
                  char pf[16];
                  char ss[16];
                  char sf[16];
                  if (sscanf (line, "%s %s %s %s", ps, pf, ss, sf) == 4)
                    {
                      i++;
                      *primary_stroke = g_realloc (*primary_stroke, i * sizeof (gchar*));
                      (*primary_stroke)[i-1] = g_strdup(ps);
                      *primary_fill = g_realloc (*primary_fill, i * sizeof (gchar*));
                      (*primary_fill)[i-1] = g_strdup(pf);
                      *secondary_stroke = g_realloc (*secondary_stroke, i * sizeof (gchar*));
                      (*secondary_stroke)[i-1] = g_strdup(ss);
                      *secondary_fill = g_realloc (*secondary_fill, i * sizeof (gchar*));
                      (*secondary_fill)[i-1] = g_strdup(sf);
                    }
                }
            }
          ptr++;
        }
      if (i > 0)
        {
          i++;
          *primary_stroke= g_realloc (*primary_stroke, i * sizeof (gchar*));
          (*primary_stroke)[i-1] = NULL;
          *primary_fill = g_realloc (*primary_fill, i * sizeof (gchar*));
          (*primary_fill)[i-1] = NULL;
          *secondary_stroke= g_realloc (*secondary_stroke, i * sizeof (gchar*));
          (*secondary_stroke)[i-1] = NULL;
          *secondary_fill = g_realloc (*secondary_fill, i * sizeof (gchar*));
          (*secondary_fill)[i-1] = NULL;
          *num_colours = i - 1;
        }
    }

  rsvg_handle_close (handle, &e);

  return 0;
}

xmlNodePtr findNode(xmlNodePtr root, const char *name)
{
  xmlNodePtr child = root->xmlChildrenNode;
  while (child) 
    {
      if (xmlStrcasecmp(child->name, BAD_CAST name) == 0)
        {
          child = child->xmlChildrenNode;
          break;
        }
      child = child->next;
    }
  return child;
}
