/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "head.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"

DoodleHashHead* doodlehash_head_new()
{
  DoodleHashHead *h = g_malloc (sizeof (DoodleHashHead));
  if (!h)
    return NULL;
  memset (h, 0, sizeof (DoodleHashHead));
  h->base = doodlehash_object_new();
  return h;
}

DoodleHashHead * doodlehash_head_copy(const DoodleHashHead*orig)
{
  DoodleHashHead *head = doodlehash_head_new();
  g_free (head->base);
  head->base = doodlehash_object_copy(orig->base);
  head->connect_to_chin_x= orig->connect_to_chin_x; 
  head->connect_to_chin_y= orig->connect_to_chin_y; 
  head->connect_to_nose_x= orig->connect_to_nose_x; 
  head->connect_to_nose_y= orig->connect_to_nose_y; 
  head->connect_to_mouth_x= orig->connect_to_mouth_x; 
  head->connect_to_mouth_y= orig->connect_to_mouth_y; 
  head->connect_to_eyes_x= orig->connect_to_eyes_x; 
  head->connect_to_eyes_y= orig->connect_to_eyes_y; 
  head->connect_to_left_ear_x= orig->connect_to_left_ear_x; 
  head->connect_to_left_ear_y= orig->connect_to_left_ear_y; 
  head->connect_to_right_ear_x= orig->connect_to_right_ear_x; 
  head->connect_to_right_ear_y= orig->connect_to_right_ear_y; 
  head->connect_to_left_cheek_x= orig->connect_to_left_cheek_x; 
  head->connect_to_left_cheek_y= orig->connect_to_left_cheek_y; 
  head->connect_to_right_cheek_x= orig->connect_to_right_cheek_x; 
  head->connect_to_right_cheek_y= orig->connect_to_right_cheek_y;
  return head;
}


int doodlehash_head_load_from_svg(DoodleHashHead *h, const char *file)
{
  if (doodlehash_object_load_from_svg_file(h->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)h->base->data, 
                                                 h->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-chin", 
                         &h->connect_to_chin_x, 
                         &h->connect_to_chin_y) != 0)
    ;//return -2;
  if (find_connect_point(handle, "#connect-to-nose", 
                         &h->connect_to_nose_x,
                         &h->connect_to_nose_y) != 0)
    ;//return -3;
  if (find_connect_point(handle, "#connect-to-mouth", 
                         &h->connect_to_mouth_x,
                         &h->connect_to_mouth_y) != 0)
    ;//return -4;
  if (find_connect_point(handle, "#connect-to-eyes", 
                         &h->connect_to_eyes_x,
                         &h->connect_to_eyes_y) != 0)
    ;//return -5;
  if (find_connect_point(handle, "#connect-to-left-ear", 
                         &h->connect_to_left_ear_x,
                         &h->connect_to_left_ear_y) != 0)
    ;//return -7;
  if (find_connect_point(handle, "#connect-to-right-ear", 
                         &h->connect_to_right_ear_x,
                         &h->connect_to_right_ear_y) != 0)
    ;//return -8;
  if (find_connect_point(handle, "#connect-to-left-cheek", 
                         &h->connect_to_left_cheek_x,
                         &h->connect_to_left_cheek_y) != 0)
    ;//return -9;
  if (find_connect_point(handle, "#connect-to-right-cheek", 
                         &h->connect_to_right_cheek_x,
                         &h->connect_to_right_cheek_y) != 0)
    ;//return -10;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashHead* doodlehash_head_load_from_svg_file(const char *file)
{
  DoodleHashHead *head = doodlehash_head_new();
  if (doodlehash_head_load_from_svg(head, file) == 0)
    return head;
  else
    {
      doodlehash_head_free(head);
      return NULL;
    }
}

void doodlehash_head_get_width_and_height(DoodleHashHead* head, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(head->base, width, height);
}

void doodlehash_head_set_width(DoodleHashHead* head, guint32 width)
{
  doodlehash_object_set_width(head->base, width);
}

void doodlehash_head_set_height(DoodleHashHead* head, guint32 height)
{
  doodlehash_object_set_height(head->base, height);
}

void doodlehash_head_get_eyes_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_eyes_x;
  *y = head->connect_to_eyes_y;
}

void doodlehash_head_set_eyes_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_eyes_x = x;
  head->connect_to_eyes_y = y;
}

void doodlehash_head_get_left_ear_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_left_ear_x;
  *y = head->connect_to_left_ear_y;
}

void doodlehash_head_set_left_ear_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_left_ear_x = x;
  head->connect_to_left_ear_y = y;
}

void doodlehash_head_get_right_ear_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_right_ear_x;
  *y = head->connect_to_right_ear_y;
}

void doodlehash_head_set_right_ear_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_right_ear_x = x;
  head->connect_to_right_ear_y = y;
}

void doodlehash_head_get_nose_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_nose_x;
  *y = head->connect_to_nose_y;
}

void doodlehash_head_set_nose_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_nose_x = x;
  head->connect_to_nose_y = y;
}

void doodlehash_head_get_mouth_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_mouth_x;
  *y = head->connect_to_mouth_y;
}

void doodlehash_head_set_mouth_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_mouth_x = x;
  head->connect_to_mouth_y = y;
}

guint32 doodlehash_head_get_width(DoodleHashHead* head)
{
  return doodlehash_object_get_width(head->base);
}

guint32 doodlehash_head_get_height(DoodleHashHead* head)
{
  return doodlehash_object_get_height(head->base);
}

guint32 doodlehash_head_get_nose_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_nose_x;
}

guint32 doodlehash_head_get_nose_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_nose_y;
}

guint32 doodlehash_head_get_eyes_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_eyes_x;
}
guint32 doodlehash_head_get_eyes_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_eyes_y;
}

guint32 doodlehash_head_get_left_ear_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_left_ear_x;
}
guint32 doodlehash_head_get_left_ear_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_left_ear_y;
}

guint32 doodlehash_head_get_right_ear_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_right_ear_x;
}

guint32 doodlehash_head_get_right_ear_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_right_ear_y;
}

guint32 doodlehash_head_get_mouth_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_mouth_x;
}

guint32 doodlehash_head_get_mouth_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_mouth_y;
}

void doodlehash_head_get_chin_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_chin_x;
  *y = head->connect_to_chin_y;
}

void doodlehash_head_set_chin_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_chin_x = x;
  head->connect_to_chin_y = y;
}

guint32 doodlehash_head_get_chin_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_chin_x;
}

guint32 doodlehash_head_get_chin_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_chin_y;
}

void doodlehash_head_get_left_cheek_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_left_cheek_x;
  *y = head->connect_to_left_cheek_y;
}

void doodlehash_head_set_left_cheek_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_left_cheek_x = x;
  head->connect_to_left_cheek_y = y;
}

void doodlehash_head_get_right_cheek_connection_point(DoodleHashHead* head, guint32* x, guint32* y)
{
  *x = head->connect_to_right_cheek_x;
  *y = head->connect_to_right_cheek_y;
}

void doodlehash_head_set_right_cheek_connection_point(DoodleHashHead* head, guint32 x, guint32 y)
{
  head->connect_to_right_cheek_x = x;
  head->connect_to_right_cheek_y = y;
}

guint32 doodlehash_head_get_left_cheek_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_left_cheek_x;
}
guint32 doodlehash_head_get_left_cheek_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_left_cheek_y;
}

guint32 doodlehash_head_get_right_cheek_connection_point_x(DoodleHashHead* head)
{
  return head->connect_to_right_cheek_x;
}

guint32 doodlehash_head_get_right_cheek_connection_point_y(DoodleHashHead* head)
{
  return head->connect_to_right_cheek_y;
}

void doodlehash_head_set_random_colours(DoodleHashHead* head, guint code)
{
  doodlehash_object_set_random_colours(head->base, code);
}

void doodlehash_head_set_colours(DoodleHashHead* head, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(head->base, code, s, f);
}

gchar *doodlehash_head_get_fill_colour(DoodleHashHead* head, guint code)
{
  return doodlehash_object_get_fill_colour(head->base, code);
}

gchar *doodlehash_head_get_stroke_colour(DoodleHashHead* head, guint code)
{
  return doodlehash_object_get_stroke_colour(head->base, code);
}

void doodlehash_head_free(DoodleHashHead* head)
{
  DoodleHashHead *h = (DoodleHashHead*) head;
  if (h && h->base)
    doodlehash_object_free (h->base);
  g_free (head);
}
