/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "ears.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include <stdlib.h>
#include "util.h"
#include "object.h"

DoodleHashEarPair* doodlehash_ears_new()
{
  DoodleHashEarPair *e = g_malloc (sizeof (DoodleHashEarPair));
  if (!e)
    return NULL;
  memset (e, 0, sizeof (DoodleHashEarPair));
  e->left = g_malloc (sizeof (DoodleHashEar));
  if (!e->left)
    {
      g_free (e);
      return NULL;
    }
  memset (e->left, 0, sizeof (DoodleHashEar));
  e->left->base = doodlehash_object_new();
  e->right = g_malloc (sizeof (DoodleHashEar));
  if (!e->right)
    {
      g_free (e->left);
      g_free (e);
      return NULL;
    }
  memset (e->right, 0, sizeof (DoodleHashEar));
  e->right->base = doodlehash_object_new();
  return e;
}
void doodlehash_ear_copy(const DoodleHashEar*orig, DoodleHashEar *copy)
{
  g_free (copy->base);
  copy->base = doodlehash_object_copy(orig->base);
  copy->connect_to_head_x = orig->connect_to_head_x;
  copy->connect_to_head_y = orig->connect_to_head_y;
  copy->connect_to_earring_x = orig->connect_to_earring_x;
  copy->connect_to_earring_y = orig->connect_to_earring_y;
}

DoodleHashEarPair* doodlehash_ears_copy(const DoodleHashEarPair *orig)
{
  DoodleHashEarPair *ears = doodlehash_ears_new();
  if (orig->left)
    doodlehash_ear_copy(orig->left, ears->left);
  if (orig->right)
    doodlehash_ear_copy(orig->right, ears->right);
  return ears;
}

int doodlehash_ear_load_from_svg(DoodleHashEar *e, const char *file)
{
  if (doodlehash_object_load_from_svg_file(e->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)e->base->data, 
                                                 e->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-head", 
                         &e->connect_to_head_x, &e->connect_to_head_y) != 0)
    return -2;
  if (find_connect_point(handle, "#connect-to-earring", 
                         &e->connect_to_earring_x, 
                         &e->connect_to_earring_y) != 0)
    return -3;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashEarPair* doodlehash_ears_load_from_svg_file(const gchar *left, const gchar *right)
{
  DoodleHashEarPair *ear = doodlehash_ears_new();
  if (doodlehash_ear_load_from_svg(ear->left, left) == 0 &&
      doodlehash_ear_load_from_svg(ear->right, right) == 0)
    return ear;
  else
    {
      doodlehash_ears_free(ear);
      return NULL;
    }
}
void doodlehash_ears_get_width_and_height(DoodleHashEarPair* ears, guint side, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base, width, height);
}

void doodlehash_ears_set_width(DoodleHashEarPair* ea, guint side, guint32 width)
{
  DoodleHashEar *e = (side == DOODLEHASH_SIDE_LEFT) ? ea->left : ea->right;
  double scale = (double) width / (double)e->base->width;
  e->connect_to_head_x *= scale;
  doodlehash_object_set_width (e->base, e->base->width * scale);
}

guint32 doodlehash_ears_get_width(DoodleHashEarPair* ears, guint side)
{
  return doodlehash_object_get_width(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base);
}

guint32 doodlehash_ears_get_height(DoodleHashEarPair* ears, guint side)
{
  return doodlehash_object_get_height(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base);
}

void doodlehash_ears_set_height(DoodleHashEarPair* ea, guint side, guint32 height)
{
  DoodleHashEar *e = (side == DOODLEHASH_SIDE_LEFT) ? ea->left : ea->right;
  double scale = (double) height / (double)e->base->height;
  e->connect_to_head_x *= scale;
  doodlehash_object_set_height (e->base, e->base->height * scale);
}

void doodlehash_ears_get_head_connection_point(DoodleHashEarPair* ears, guint side, guint32* x, guint32* y)
{
  DoodleHashEar *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = ears->left;
  else
    e = ears->right;
  *x = e->connect_to_head_x;
  *y = e->connect_to_head_y;
}

void doodlehash_ears_set_head_connection_point(DoodleHashEarPair* ears, guint side, guint32 x, guint32 y)
{
  DoodleHashEar *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = ears->left;
  else
    e = ears->right;
  e->connect_to_head_x = x;
  e->connect_to_head_y = y;
}

guint32 doodlehash_ears_get_head_connection_point_x(DoodleHashEarPair* ears, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? ears->left->connect_to_head_x : ears->right->connect_to_head_x;
}

guint32 doodlehash_ears_get_head_connection_point_y(DoodleHashEarPair* ears, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? ears->left->connect_to_head_y : ears->right->connect_to_head_y;
}

void doodlehash_ears_get_earring_connection_point(DoodleHashEarPair* ears, guint side, guint32* x, guint32* y)
{
  DoodleHashEar *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = ears->left;
  else
    e = ears->right;
  *x = e->connect_to_earring_x;
  *y = e->connect_to_earring_y;
}

void doodlehash_ears_set_earring_connection_point(DoodleHashEarPair* ears, guint side, guint32 x, guint32 y)
{
  DoodleHashEar *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = ears->left;
  else
    e = ears->right;
  e->connect_to_earring_x = x;
  e->connect_to_earring_y = y;
}

guint32 doodlehash_ears_get_earring_connection_point_x(DoodleHashEarPair* ears, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? ears->left->connect_to_earring_x : ears->right->connect_to_earring_x;
}

guint32 doodlehash_ears_get_earring_connection_point_y(DoodleHashEarPair* ears, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? ears->left->connect_to_earring_y : ears->right->connect_to_earring_y;
}

void doodlehash_ears_set_random_colours(DoodleHashEarPair* ears, guint code, int matching)
{
  doodlehash_object_set_random_matching_colours(ears->left->base, 
                                                ears->right->base, 
                                                code, matching);
}

void doodlehash_ears_set_colours(DoodleHashEarPair* ears, guint side, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base, code, s, f);
}

gchar *doodlehash_ears_get_fill_colour(DoodleHashEarPair* ears, guint side, guint code)
{
  return doodlehash_object_get_fill_colour(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base, code);
}

gchar *doodlehash_ears_get_stroke_colour(DoodleHashEarPair* ears, guint side, guint code)
{
  return doodlehash_object_get_stroke_colour(side == DOODLEHASH_SIDE_LEFT ? ears->left->base : ears->right->base, code);
}

void doodlehash_ears_scale(DoodleHashEarPair *ea, guint side, double scale)
{
  DoodleHashEar *e = (side == DOODLEHASH_SIDE_LEFT) ? ea->left : ea->right;
  e->connect_to_head_x *= scale;
  e->connect_to_head_y *= scale;
  doodlehash_object_scale(e->base, scale);
}

void doodlehash_ears_free(DoodleHashEarPair* e)
{
  if (e && e->left && e->left->base)
    doodlehash_object_free(e->left->base);
  if (e && e->right && e->right->base)
    doodlehash_object_free(e->right->base);
  if (e && e->right)
    g_free (e->right);
  if (e && e->left)
    g_free (e->left);
  g_free (e);
}

