/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include <string.h>
#include <stdlib.h>
#include <librsvg/rsvg.h>
#include "doodlehash.h"
#include "face.h"
#include "util.h"
#include "object.h"
#include "loader.h"

int gdk_pixmap_get_width(GdkPixmap *pixmap)
{
  int width;
  int height;
  gdk_pixmap_get_size(pixmap, &width, &height);
  return width;
}
int gdk_pixmap_get_height(GdkPixmap *pixmap)
{
  int width;
  int height;
  gdk_pixmap_get_size(pixmap, &width, &height);
  return height;
}
double get_aspect_ratio(guint32 inner_width, guint32 inner_height, guint32 width, guint32 height)
{
  guint32 canvas_width = width;
  guint32 canvas_height = height;
  double aspect = 0;
  double aspect_width =  (double)canvas_width / (double)inner_width;
  double aspect_height = (double)canvas_height / (double)inner_height;

  if (aspect_width < aspect_height)
    aspect = aspect_width;
  else
    aspect = aspect_height;
  return aspect;
}

int get_random_file(const gchar *subdir, const gchar *ext, gchar **file)
{
  gchar * path = get_svg_path (subdir);
  GDir *dir = g_dir_open (path, 0, NULL);
  size_t num_files = 0;
  if (dir)
    {
      const char *basename;
      const char **files = NULL;
      while ((basename = g_dir_read_name (dir)) != NULL)
        {
          if (g_str_has_suffix(basename, ext))
            {
              num_files++;
              files = g_realloc (files, sizeof(const char **) * num_files);
              files[num_files-1] = basename;
            }
        }
      if (num_files > 0)
        *file = g_build_filename (DOODLEHASH_DATADIR, subdir, files[rand() % num_files], NULL);
      g_dir_close(dir);
    }
  g_free (path);
  return dir && num_files ? 0 : 1;
}

int get_random_file_pair(const gchar *subdir, const gchar *ext, gchar **left, gchar **right)
{
  int retval = 1;
  gchar *left_ext = g_strdup_printf("%s%s", "-left", ext);
  if (left_ext)
    {
      retval = get_random_file(subdir, left_ext, left);
      if (retval == 0)
        {
          gchar *base = g_path_get_basename(*left);
          if (base)
            {
              base[strlen(base)-strlen(left_ext)] = 0;
              gchar *right_ext = g_strdup_printf("%s%s%s", base, "-right",ext);
              if (right_ext)
                {
                  retval = get_random_file(subdir, right_ext, right);
                  g_free(right_ext);
                }
              g_free(base);
            }
        }
      g_free (left_ext);
    }
  return retval;
}

size_t get_num_from_hash(const gchar *hash)
{
  size_t sum = 0;
  const gchar *letter;
  for (letter = &hash[0]; *letter; letter++)
    sum += *letter +rand();
  return sum;
}

void resize_svg (gint * width, gint * height, gpointer user_data)
{
  RsvgPositionData *dim = (RsvgPositionData*)user_data;
  *width = dim->x;
  *height = dim->y;
}

GdkPixbuf* load_pixbuf(DoodleHashObject *o, guint32 width, guint32 height)
{
  /* where is gdk_new_pixbuf_from_file_data_at_size? */
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)o->data, o->data_len, NULL);
  if (!handle)
    return NULL;

  RsvgPositionData dim;
  dim.x = width;
  dim.y = height;
  rsvg_handle_set_size_callback (handle, resize_svg, &dim, NULL);
  GdkPixbuf *p = rsvg_handle_get_pixbuf(handle);

  rsvg_handle_close (handle, NULL);
  return p;
}

void draw_nose_on_head(DoodleHashNose *nose, DoodleHashHead *head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_nose_get_width(nose) * (double)percent;
  guint32 new_height = (double)doodlehash_nose_get_height(nose) * (double)percent;
  GdkPixbuf *nose_pixbuf = load_pixbuf(nose->base, new_width, new_height);
  guint32 new_nose_x = doodlehash_nose_get_head_connection_point_x(nose) * percent;
  guint32 new_nose_y = doodlehash_nose_get_head_connection_point_y(nose) * percent;
  guint32 new_head_x = doodlehash_head_get_nose_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_nose_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_nose_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_nose_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, nose_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (nose_pixbuf);
}

void draw_chin_on_head(DoodleHashChin *chin, DoodleHashHead *head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_chin_get_width(chin) * (double)percent;
  guint32 new_height = (double)doodlehash_chin_get_height(chin) * (double)percent;
  GdkPixbuf *chin_pixbuf = load_pixbuf(chin->base, new_width, new_height);
  guint32 new_chin_x = doodlehash_chin_get_head_connection_point_x(chin) * percent;
  guint32 new_chin_y = doodlehash_chin_get_head_connection_point_y(chin) * percent;
  guint32 new_head_x = doodlehash_head_get_chin_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_chin_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_chin_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_chin_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, chin_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (chin_pixbuf);
}

void draw_eyes_on_head(DoodleHashEyes *eye, DoodleHashHead *head, GdkPixmap*pixmap)
{
  //just like drawing a chin.
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_eyes_get_width(eye) * (double)percent;
  guint32 new_height = (double)doodlehash_eyes_get_height(eye) * (double)percent;
  GdkPixbuf *eyes_pixbuf = load_pixbuf(eye->base, new_width, new_height);
  guint32 new_eyes_x = doodlehash_eyes_get_head_connection_point_x(eye) * percent;
  guint32 new_eyes_y = doodlehash_eyes_get_head_connection_point_y(eye) * percent;
  guint32 new_head_x = doodlehash_head_get_eyes_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_eyes_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_eyes_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_eyes_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, eyes_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (eyes_pixbuf);
}

void draw_cheeks_on_head(DoodleHashCheekPair *cheek, DoodleHashHead *head, GdkPixmap*pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_cheeks_get_width(cheek, DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_height = (double)doodlehash_cheeks_get_height(cheek, DOODLEHASH_SIDE_LEFT) * (double)percent;
  GdkPixbuf *cheek_pixbuf = load_pixbuf(cheek->left->base, new_width, new_height);
  guint32 new_cheek_x = doodlehash_cheeks_get_head_connection_point_x(cheek, DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_cheek_y = doodlehash_cheeks_get_head_connection_point_y(cheek, DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_head_x = doodlehash_head_get_left_cheek_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_left_cheek_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_cheek_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_cheek_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, cheek_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (cheek_pixbuf);
  //now we do the right cheek
  new_width = (double)doodlehash_cheeks_get_width(cheek, DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_height = (double)doodlehash_cheeks_get_height(cheek, DOODLEHASH_SIDE_RIGHT) * (double)percent;
  cheek_pixbuf = load_pixbuf(cheek->right->base, new_width, new_height);
  new_cheek_x = doodlehash_cheeks_get_head_connection_point_x(cheek, DOODLEHASH_SIDE_RIGHT) * percent;
  new_cheek_y = doodlehash_cheeks_get_head_connection_point_y(cheek, DOODLEHASH_SIDE_RIGHT) * percent;
  new_head_x = doodlehash_head_get_right_cheek_connection_point_x(head) * percent;
  new_head_y = doodlehash_head_get_right_cheek_connection_point_y(head) * percent;
  delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_cheek_x);
  delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_cheek_y);

  head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, cheek_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (cheek_pixbuf);
}


void draw_mouth_on_head(DoodleHashMouth *mouth, DoodleHashHead *head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_mouth_get_width(mouth) * (double)percent;
  guint32 new_height = (double)doodlehash_mouth_get_height(mouth) * (double)percent;
  GdkPixbuf *mouth_pixbuf = load_pixbuf(mouth->base, new_width, new_height);
  guint32 new_mouth_x = doodlehash_mouth_get_head_connection_point_x(mouth) * percent;
  guint32 new_mouth_y = doodlehash_mouth_get_head_connection_point_y(mouth) * percent;
  guint32 new_head_x = doodlehash_head_get_mouth_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_mouth_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_mouth_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_mouth_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, mouth_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (mouth_pixbuf);
}

void draw_moustache_on_mouth(DoodleHashMoustache* moustache, DoodleHashMouth* mouth, DoodleHashHead* head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_mouth_width = (double)doodlehash_mouth_get_width(mouth) * (double)percent;
  guint32 new_mouth_height = (double)doodlehash_mouth_get_height(mouth) * (double)percent;
  guint32 mouth_x = doodlehash_mouth_get_head_connection_point_x(mouth) * percent;
  guint32 mouth_y = doodlehash_mouth_get_head_connection_point_y(mouth) * percent;
  guint32 new_head_x = doodlehash_head_get_mouth_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_mouth_connection_point_y(head) * percent;
  int delta_x = (new_head_x-(new_mouth_width/2))+ ((new_mouth_width/2)-mouth_x);
  int delta_y = (new_head_y-(new_mouth_height/2)) + ((new_mouth_height/2)-mouth_y);

  guint32 new_width = (double)doodlehash_moustache_get_width(moustache) * (double)percent;
  guint32 new_height = (double)doodlehash_moustache_get_height(moustache) * (double)percent;
  guint32 new_moustache_x = doodlehash_moustache_get_mouth_connection_point_x(moustache) * percent;
  guint32 new_moustache_y = doodlehash_moustache_get_mouth_connection_point_y(moustache) * percent;
  guint32 new_mouth_x = doodlehash_mouth_get_moustache_connection_point_x(mouth) * percent;
  guint32 new_mouth_y = doodlehash_mouth_get_moustache_connection_point_y(mouth) * percent;
  int more_delta_x = (new_mouth_x-(new_width/2))+ ((new_width/2)-new_moustache_x);
  int more_delta_y = (new_mouth_y-(new_height/2)) + ((new_height/2)-new_moustache_y);

  GdkPixbuf *moustache_pixbuf = load_pixbuf(moustache->base, new_width, new_height);
  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, moustache_pixbuf, 
                               head_canvas_offset_x + delta_x + more_delta_x,
                               head_canvas_offset_y + delta_y + more_delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (moustache_pixbuf);
}

void draw_eyebrows_on_eyes(DoodleHashEyeBrowPair* eyebrow, DoodleHashEyes* eye, DoodleHashHead* head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_eye_width = (double)doodlehash_eyes_get_width(eye) * (double)percent;
  guint32 new_eye_height = (double)doodlehash_eyes_get_height(eye) * (double)percent;
  guint32 eye_x = doodlehash_eyes_get_head_connection_point_x(eye) * percent;
  guint32 eye_y = doodlehash_eyes_get_head_connection_point_y(eye) * percent;
  guint32 new_head_x = doodlehash_head_get_eyes_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_eyes_connection_point_y(head) * percent;
  int delta_x = (new_head_x-(new_eye_width/2))+ ((new_eye_width/2)-eye_x);
  int delta_y = (new_head_y-(new_eye_height/2)) + ((new_eye_height/2)-eye_y);

  guint32 new_width = (double)doodlehash_eyebrows_get_width(eyebrow,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_height = (double)doodlehash_eyebrows_get_height(eyebrow,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_eyebrow_x = doodlehash_eyebrows_get_eye_connection_point_x(eyebrow,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_eyebrow_y = doodlehash_eyebrows_get_eye_connection_point_y(eyebrow,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_eye_x = doodlehash_eyes_get_left_brow_connection_point_x(eye) * percent;
  guint32 new_eye_y = doodlehash_eyes_get_left_brow_connection_point_y(eye) * percent;
  int more_delta_x = (new_eye_x-(new_width/2))+ ((new_width/2)-new_eyebrow_x);
  int more_delta_y = (new_eye_y-(new_height/2)) + ((new_height/2)-new_eyebrow_y);

  GdkPixbuf *eyebrow_pixbuf = load_pixbuf(eyebrow->left->base, new_width, new_height);
  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, eyebrow_pixbuf, 
                               head_canvas_offset_x + delta_x + more_delta_x,
                               head_canvas_offset_y + delta_y + more_delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (eyebrow_pixbuf);
  new_eye_width = (double)doodlehash_eyes_get_width(eye) * (double)percent;
  new_eye_height = (double)doodlehash_eyes_get_height(eye) * (double)percent;
  eye_x = doodlehash_eyes_get_head_connection_point_x(eye) * percent;
  eye_y = doodlehash_eyes_get_head_connection_point_y(eye) * percent;
  new_head_x = doodlehash_head_get_eyes_connection_point_x(head) * percent;
  new_head_y = doodlehash_head_get_eyes_connection_point_y(head) * percent;
  delta_x = (new_head_x-(new_eye_width/2))+ ((new_eye_width/2)-eye_x);
  delta_y = (new_head_y-(new_eye_height/2)) + ((new_eye_height/2)-eye_y);

  new_width = (double)doodlehash_eyebrows_get_width(eyebrow,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_height = (double)doodlehash_eyebrows_get_height(eyebrow,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_eyebrow_x = doodlehash_eyebrows_get_eye_connection_point_x(eyebrow,DOODLEHASH_SIDE_RIGHT) * percent;
  new_eyebrow_y = doodlehash_eyebrows_get_eye_connection_point_y(eyebrow,DOODLEHASH_SIDE_RIGHT) * percent;
  new_eye_x = doodlehash_eyes_get_right_brow_connection_point_x(eye) * percent;
  new_eye_y = doodlehash_eyes_get_right_brow_connection_point_y(eye) * percent;
  more_delta_x = (new_eye_x-(new_width/2))+ ((new_width/2)-new_eyebrow_x);
  more_delta_y = (new_eye_y-(new_height/2)) + ((new_height/2)-new_eyebrow_y);

  eyebrow_pixbuf = load_pixbuf(eyebrow->right->base, new_width, new_height);
  head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, eyebrow_pixbuf, 
                               head_canvas_offset_x + delta_x + more_delta_x,
                               head_canvas_offset_y + delta_y + more_delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (eyebrow_pixbuf);
}

void draw_ears_on_head(DoodleHashEarPair* ear, DoodleHashHead* head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_ears_get_width(ear, DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_height = (double)doodlehash_ears_get_height(ear, DOODLEHASH_SIDE_LEFT) * (double)percent;
  GdkPixbuf *ear_pixbuf = load_pixbuf(ear->left->base, new_width, new_height);
  guint32 new_ear_x = doodlehash_ears_get_head_connection_point_x(ear, DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_ear_y = doodlehash_ears_get_head_connection_point_y(ear, DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_head_x = doodlehash_head_get_left_ear_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_left_ear_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_ear_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_ear_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, ear_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (ear_pixbuf);
  //now we do the right ear

  new_width = (double)doodlehash_ears_get_width(ear, DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_height = (double)doodlehash_ears_get_height(ear, DOODLEHASH_SIDE_RIGHT) * (double)percent;
  ear_pixbuf = load_pixbuf(ear->right->base, new_width, new_height);
  new_ear_x = doodlehash_ears_get_head_connection_point_x(ear, DOODLEHASH_SIDE_RIGHT) * percent;
  new_ear_y = doodlehash_ears_get_head_connection_point_y(ear, DOODLEHASH_SIDE_RIGHT) * percent;
  new_head_x = doodlehash_head_get_right_ear_connection_point_x(head) * percent;
  new_head_y = doodlehash_head_get_right_ear_connection_point_y(head) * percent;
  delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_ear_x);
  delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_ear_y);

  head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, ear_pixbuf, 
                               head_canvas_offset_x + delta_x,
                               head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (ear_pixbuf);
}

void draw_earrings_on_ears(DoodleHashEarRingPair* earring, DoodleHashEarPair* ear, DoodleHashHead* head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_ear_width = (double)doodlehash_ears_get_width(ear,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_ear_height = (double)doodlehash_ears_get_height(ear,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 ear_x = doodlehash_ears_get_head_connection_point_x(ear,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 ear_y = doodlehash_ears_get_head_connection_point_y(ear,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_head_x = doodlehash_head_get_left_ear_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_left_ear_connection_point_y(head) * percent;
  int delta_x = (new_head_x-(new_ear_width/2))+ ((new_ear_width/2)-ear_x);
  int delta_y = (new_head_y-(new_ear_height/2)) + ((new_ear_height/2)-ear_y);

  guint32 new_width = (double)doodlehash_earrings_get_width(earring,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_height = (double)doodlehash_earrings_get_height(earring,DOODLEHASH_SIDE_LEFT) * (double)percent;
  guint32 new_earring_x = doodlehash_earrings_get_ear_connection_point_x(earring,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_earring_y = doodlehash_earrings_get_ear_connection_point_y(earring,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_ear_x = doodlehash_ears_get_earring_connection_point_x(ear,DOODLEHASH_SIDE_LEFT) * percent;
  guint32 new_ear_y = doodlehash_ears_get_earring_connection_point_y(ear,DOODLEHASH_SIDE_LEFT) * percent;
  int more_delta_x = (new_ear_x-(new_width/2))+ ((new_width/2)-new_earring_x);
  int more_delta_y = (new_ear_y-(new_height/2)) + ((new_height/2)-new_earring_y);

  GdkPixbuf *earring_pixbuf = load_pixbuf(earring->left->base, new_width, new_height);
  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, earring_pixbuf, 
                               head_canvas_offset_x + delta_x + more_delta_x,
                               head_canvas_offset_y + delta_y + more_delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (earring_pixbuf);
  new_ear_width = (double)doodlehash_ears_get_width(ear,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_ear_height = (double)doodlehash_ears_get_height(ear,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  ear_x = doodlehash_ears_get_head_connection_point_x(ear,DOODLEHASH_SIDE_RIGHT) * percent;
  ear_y = doodlehash_ears_get_head_connection_point_y(ear,DOODLEHASH_SIDE_RIGHT) * percent;
  new_head_x = doodlehash_head_get_right_ear_connection_point_x(head) * percent;
  new_head_y = doodlehash_head_get_right_ear_connection_point_y(head) * percent;
  delta_x = (new_head_x-(new_ear_width/2))+ ((new_ear_width/2)-ear_x);
  delta_y = (new_head_y-(new_ear_height/2)) + ((new_ear_height/2)-ear_y);

  new_width = (double)doodlehash_earrings_get_width(earring,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_height = (double)doodlehash_earrings_get_height(earring,DOODLEHASH_SIDE_RIGHT) * (double)percent;
  new_earring_x = doodlehash_earrings_get_ear_connection_point_x(earring,DOODLEHASH_SIDE_RIGHT) * percent;
  new_earring_y = doodlehash_earrings_get_ear_connection_point_y(earring,DOODLEHASH_SIDE_RIGHT) * percent;
  new_ear_x = doodlehash_ears_get_earring_connection_point_x(ear,DOODLEHASH_SIDE_RIGHT) * percent;
  new_ear_y = doodlehash_ears_get_earring_connection_point_y(ear,DOODLEHASH_SIDE_RIGHT) * percent;
  more_delta_x = (new_ear_x-(new_width/2))+ ((new_width/2)-new_earring_x);
  more_delta_y = (new_ear_y-(new_height/2)) + ((new_height/2)-new_earring_y);

  earring_pixbuf = load_pixbuf(earring->right->base, new_width, new_height);
  head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, earring_pixbuf, 
                               head_canvas_offset_x + delta_x + more_delta_x,
                               head_canvas_offset_y + delta_y + more_delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (earring_pixbuf);
}

void draw_spectacles_on_head(DoodleHashSpectacles* spectacles, DoodleHashEyes *eye, DoodleHashHead* head, GdkPixmap *pixmap)
{
  //drawn exactly like eyes, but with the connection points of eyes.
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_eyes_get_width(eye) * (double)percent;
  guint32 new_height = (double)doodlehash_eyes_get_height(eye) * (double)percent;
  GdkPixbuf *spectacles_pixbuf = load_pixbuf(spectacles->base, new_width, new_height);
  guint32 new_eyes_x = doodlehash_eyes_get_head_connection_point_x(eye) * percent;
  guint32 new_eyes_y = doodlehash_eyes_get_head_connection_point_y(eye) * percent;
  guint32 new_head_x = doodlehash_head_get_eyes_connection_point_x(head) * percent;
  guint32 new_head_y = doodlehash_head_get_eyes_connection_point_y(head) * percent;
  int delta_x = new_head_x-(new_width/2)+ ((new_width/2)-new_eyes_x);
  int delta_y = new_head_y-(new_height/2) + ((new_height/2)-new_eyes_y);

  guint32 head_canvas_offset_x = ((gdk_pixmap_get_width(pixmap) - (doodlehash_head_get_width(head)*percent)) / 2);
  guint32 head_canvas_offset_y = ((gdk_pixmap_get_height(pixmap) - (doodlehash_head_get_height(head)*percent)) / 2);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf
    (cr, spectacles_pixbuf, head_canvas_offset_x + delta_x,
     head_canvas_offset_y + delta_y);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (spectacles_pixbuf);
}

void draw_hair_on_head(DoodleHashHair* hair, DoodleHashHead* head, GdkPixmap *pixmap)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), gdk_pixmap_get_width(pixmap), gdk_pixmap_get_height(pixmap));
  guint32 new_width = (double)doodlehash_hair_get_width(hair) * (double)percent;
  guint32 new_height = (double)doodlehash_hair_get_height(hair) * (double)percent;
  GdkPixbuf *hair_pixbuf = load_pixbuf(hair->base, new_width, new_height);
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf (cr, hair_pixbuf, 0, 0);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (hair_pixbuf);
}

void draw_head(DoodleHashHead* head, GdkPixmap *pixmap, guint32 width, guint32 height)
{
  double percent = get_aspect_ratio(doodlehash_head_get_width(head), doodlehash_head_get_height(head), width, height);
  guint32 new_width = (double)doodlehash_head_get_width(head) * (double)percent;
  guint32 new_height = (double)doodlehash_head_get_height(head) * (double)percent;
  GdkPixbuf *head_pixbuf = load_pixbuf(head->base, new_width, new_height);
  int delta_x = width - new_width;
  int delta_y = height - new_height;
  cairo_t *cr = gdk_cairo_create(pixmap);
  gdk_cairo_set_source_pixbuf(cr, head_pixbuf, delta_x/2, delta_y/2);
  cairo_paint(cr);
  cairo_destroy(cr);
  g_object_unref (head_pixbuf);
}

int get_same_width_svg_file(const char *infile, const char *subdir, const char *ext, gchar **file)
{
  guint32 width = doodlehash_object_get_width_from_svg_file(infile);
  if (width == 0)
    return -1;
  gchar * path = get_svg_path (subdir);
  GDir *dir = g_dir_open (path, 0, NULL);
  size_t num_files = 0;
  if (dir)
    {
      const char *basename;
      const char **files = NULL;
      while ((basename = g_dir_read_name (dir)) != NULL)
        {
          *file = g_build_filename (DOODLEHASH_DATADIR, subdir, basename, NULL);
          if (g_str_has_suffix(basename, ext) &&
              doodlehash_object_get_width_from_svg_file(*file) == width)
            {
              num_files++;
              files = g_realloc (files, sizeof(const char **) * num_files);
              files[num_files-1] = basename;
            }
          g_free(*file);
          *file = NULL;
        }
      if (num_files > 0)
        *file = g_build_filename (DOODLEHASH_DATADIR, subdir, files[rand() % num_files], NULL);
      g_dir_close(dir);
    }
  g_free (path);
  return dir && num_files ? 0 : 1;
}

GdkPixbuf *draw_face(DoodleHashFace *face, guint32 height, guint32 width)
{
  GdkPixmap *pixmap = gdk_pixmap_new (gdk_get_default_root_window(), width, height, -1);
  if (pixmap)
    {
      DoodleHashHead *head = doodlehash_face_get_head(face);
      DoodleHashCheekPair *cheeks = doodlehash_face_get_cheeks(face);
      DoodleHashNose *nose = doodlehash_face_get_nose(face);
      DoodleHashEyes *eyes= doodlehash_face_get_eyes(face);
      DoodleHashEyeBrowPair *eyebrows = doodlehash_face_get_eyebrows(face);
      DoodleHashSpectacles *spectacles = doodlehash_face_get_spectacles(face);
      DoodleHashChin *chin = doodlehash_face_get_chin(face);
      DoodleHashMouth *mouth = doodlehash_face_get_mouth(face);
      DoodleHashMoustache *moustache = doodlehash_face_get_moustache(face);
      DoodleHashEarPair *ears = doodlehash_face_get_ears(face);
      DoodleHashEarRingPair *earrings = doodlehash_face_get_earrings(face);
      DoodleHashHair *hair = doodlehash_face_get_hair(face);

      cairo_t *cr = gdk_cairo_create(pixmap);
      cairo_set_source_rgb (cr, 1, 1, 1);
      cairo_rectangle (cr, 0, 0, width, height);
      cairo_paint(cr);
      cairo_destroy(cr);

      if (head)
        {
          draw_head(head, pixmap, width, height);
          if (cheeks)
            draw_cheeks_on_head(cheeks, head, pixmap);
          if (nose)
            draw_nose_on_head (nose, head, pixmap);
          if (eyes)
            {
              draw_eyes_on_head (eyes, head, pixmap);
              if (eyebrows)
                draw_eyebrows_on_eyes (eyebrows, eyes, head, pixmap);
            }
          if (spectacles && eyes)
              draw_spectacles_on_head (spectacles, eyes, head, pixmap);
          if (chin)
            draw_chin_on_head(chin, head, pixmap);
          if (mouth)
            {
              draw_mouth_on_head (mouth, head, pixmap);
              if (moustache)
                draw_moustache_on_mouth (moustache, mouth, head, pixmap);
            }
          if (ears)
            {
              draw_ears_on_head(ears, head, pixmap);
              if (earrings)
                draw_earrings_on_ears(earrings, ears, head, pixmap);
            }
          if (hair)
            draw_hair_on_head(hair, head, pixmap);
        }
    }
  return gdk_pixbuf_get_from_drawable(NULL, pixmap, 
                                      gdk_drawable_get_colormap(pixmap), 
                                      0, 0, 0, 0, width, height);
}

GdkPixbuf* doodlehash_new_pixbuf(const gchar *hash, guint32 height, guint32 width)
{
  GdkPixbuf *pixbuf = NULL;
  gchar *ear_left_file = NULL;
  gchar *ear_right_file = NULL;
  gchar *earring_left_file = NULL;
  gchar *earring_right_file = NULL;
  gchar *eyes_file = NULL;
  gchar *hair_file = NULL;
  gchar *head_file = NULL;
  gchar *moustache_file = NULL;
  gchar *mouth_file = NULL;
  gchar *nose_file = NULL;
  gchar *eyebrow_left_file = NULL;
  gchar *eyebrow_right_file = NULL;
  gchar *chin_file = NULL;
  gchar *cheek_left_file = NULL;
  gchar *cheek_right_file = NULL;
  gchar *spectacles_file = NULL;
  srand(get_num_from_hash(hash));
  get_random_file_pair("ears", ".svg", &ear_left_file, &ear_right_file);
  get_random_file_pair("earrings", ".svg", &earring_left_file, &earring_right_file);
  get_random_file("eyes", ".svg", &eyes_file);
  get_random_file("heads", ".svg", &head_file);
  get_random_file("moustaches", ".svg", &moustache_file);
  get_random_file("mouths", ".svg", &mouth_file);
  get_random_file("noses", ".svg", &nose_file);
  get_random_file_pair ("eyebrows", ".svg", &eyebrow_left_file, &eyebrow_right_file);
  get_random_file("chins", ".svg", &chin_file);
  get_random_file_pair ("cheeks", ".svg", &cheek_left_file, &cheek_right_file);
  get_same_width_svg_file(head_file, "hair", ".svg", &hair_file);
  get_same_width_svg_file(eyes_file, "spectacles", ".svg", &spectacles_file);
  //printf("using these files:\n"
         //"%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n"
         //"%s\n" "%s\n" "%s\n" "%s\n" "%s\n" "%s\n",
         //ear_left_file, ear_right_file, earring_left_file, earring_right_file,
         //eyes_file, hair_file, head_file, moustache_file, mouth_file, nose_file,
         //eyebrow_left_file, eyebrow_right_file, chin_file, cheek_left_file, cheek_right_file, spectacles_file);
  DoodleHashFace *face = doodlehash_face_load_from_svg_files
    (ear_left_file, ear_right_file, earring_left_file, earring_right_file, eyes_file, hair_file, head_file, moustache_file, mouth_file, nose_file, eyebrow_left_file, eyebrow_right_file, chin_file, cheek_left_file, cheek_right_file, spectacles_file);

  doodlehash_face_set_random_colours(face);

  pixbuf = draw_face(face, height, width);
  doodlehash_face_free (face);

  g_free(ear_left_file);
  g_free(ear_right_file);
  g_free(earring_left_file);
  g_free(earring_right_file);
  g_free(eyes_file);
  g_free(hair_file);
  g_free(head_file);
  g_free(moustache_file);
  g_free(mouth_file);
  g_free(nose_file);
  g_free(eyebrow_left_file);
  g_free(eyebrow_right_file);
  g_free(chin_file);
  g_free(cheek_left_file);
  g_free(cheek_right_file);
  g_free(spectacles_file);
  return pixbuf;
}

size_t count_svg_files(gchar *subdir)
{
  char *ext = ".svg";
  size_t count = 0;
  gchar * path = get_svg_path (subdir);
  GDir *dir = g_dir_open (path, 0, NULL);
  if (dir)
    {
      const char *basename;
      while ((basename = g_dir_read_name (dir)) != NULL)
        {
          if (ext && strlen (basename) >= strlen (ext))
            {
              if (g_strncasecmp (&basename[strlen(basename)-(strlen(ext))], ext, strlen(ext)) != 0)
                continue;
              count++;
            }
        }
      g_dir_close(dir);
    }
  g_free (path);
  return count;
}

size_t doodlehash_count_svg_files()
{
  size_t count = count_svg_files("heads");
  count += count_svg_files("hair");
  count += count_svg_files("ears");
  count += count_svg_files("earrings");
  count += count_svg_files("eyes");
  count += count_svg_files("eyebrows");
  count += count_svg_files("noses");
  count += count_svg_files("mouths");
  count += count_svg_files("moustaches");
  count += count_svg_files("chins");
  count += count_svg_files("cheeks");
  count += count_svg_files("spectacles");
  return count;
}

GdkPixbuf *doodlehash_new_pixbuf_from_file(const gchar *file, guint32 height, guint32 width)
{
  GdkPixbuf * pixbuf = NULL;
  DoodleHashFace *face = doodlehash_load_face(file);
  if (face)
    {
      pixbuf = draw_face(face, height, width);
      doodlehash_face_free(face);
    }

  return pixbuf;
}

int doodlehash_init()
{
  if (doodlehash_count_svg_files() == 0)
    return -1;
  rsvg_init();
  return 0;
}

void doodlehash_term()
{
  rsvg_term();
}

