/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "cheeks.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"

DoodleHashCheekPair* doodlehash_cheeks_new()
{
  DoodleHashCheekPair *e = g_malloc (sizeof (DoodleHashCheekPair));
  if (!e)
    return NULL;
  memset (e, 0, sizeof (DoodleHashCheekPair));
  e->left = g_malloc (sizeof (DoodleHashCheek));
  if (!e->left)
    {
      g_free (e);
      return NULL;
    }
  memset (e->left, 0, sizeof (DoodleHashCheek));
  e->left->base = doodlehash_object_new();
  e->right = g_malloc (sizeof (DoodleHashCheek));
  if (!e->right)
    {
      g_free (e->left);
      g_free (e);
      return NULL;
    }
  memset (e->right, 0, sizeof (DoodleHashCheek));
  e->right->base = doodlehash_object_new();
  return e;
}

void doodlehash_cheek_copy(const DoodleHashCheek*orig, DoodleHashCheek *copy)
{
  g_free (copy->base);
  copy->base = doodlehash_object_copy(orig->base);
  copy->connect_to_head_x = orig->connect_to_head_x;
  copy->connect_to_head_y = orig->connect_to_head_y;
}

DoodleHashCheekPair* doodlehash_cheeks_copy(const DoodleHashCheekPair *orig)
{
  DoodleHashCheekPair *cheeks = doodlehash_cheeks_new();
  if (orig->left)
    doodlehash_cheek_copy(orig->left, cheeks->left);
  if (orig->right)
    doodlehash_cheek_copy(orig->right, cheeks->right);
  return cheeks;
}

int doodlehash_cheek_load_from_svg(DoodleHashCheek *c, const char *file)
{
  if (doodlehash_object_load_from_svg_file(c->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)c->base->data, 
                                                 c->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-head", 
                         &c->connect_to_head_x, &c->connect_to_head_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashCheekPair* doodlehash_cheeks_load_from_svg_file(const gchar *left, const gchar *right)
{
  DoodleHashCheekPair *cheek = doodlehash_cheeks_new();
  if (doodlehash_cheek_load_from_svg(cheek->left, left) == 0 &&
      doodlehash_cheek_load_from_svg(cheek->right, right) == 0)
    return cheek;
  else
    {
      doodlehash_cheeks_free(cheek);
      return NULL;
    }
}
void doodlehash_cheeks_get_width_and_height(DoodleHashCheekPair* cheek, guint side, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height (side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base, width, height);
}

void doodlehash_cheeks_set_width(DoodleHashCheekPair *cheek, guint side, guint32 width)
{
  DoodleHashCheek *c = (side == DOODLEHASH_SIDE_LEFT) ? cheek->left : cheek->right;
  double scale = (double) width / (double)c->base->width;
  c->connect_to_head_x *= scale;
  doodlehash_object_set_width (c->base, c->base->width * scale);
}

void doodlehash_cheeks_set_height(DoodleHashCheekPair *cheek, guint side, guint32 height)
{
  DoodleHashCheek *c = (side == DOODLEHASH_SIDE_LEFT) ? cheek->left : cheek->right;
  double scale = (double) height / (double)c->base->height;
  c->connect_to_head_x *= scale;
  doodlehash_object_set_height (c->base, c->base->height * scale);
}

void doodlehash_cheeks_get_head_connection_point(DoodleHashCheekPair *cheek, guint side, guint32* x, guint32* y)
{
  DoodleHashCheek *c;
  if (side == DOODLEHASH_SIDE_LEFT)
    c = cheek->left;
  else
    c = cheek->right;
  *x = c->connect_to_head_x;
  *y = c->connect_to_head_y;
}

void doodlehash_cheeks_set_head_connection_point(DoodleHashCheekPair *cheek, guint side, guint32 x, guint32 y)
{
  DoodleHashCheek *c;
  if (side == DOODLEHASH_SIDE_LEFT)
    c = cheek->left;
  else
    c = cheek->right;
  c->connect_to_head_x = x;
  c->connect_to_head_y = y;
}

guint32 doodlehash_cheeks_get_head_connection_point_x(DoodleHashCheekPair *cheek, guint side)
{
  if (side == DOODLEHASH_SIDE_LEFT)
    return cheek->left->connect_to_head_x;
  else
    return cheek->right->connect_to_head_x;
}

guint32 doodlehash_cheeks_get_head_connection_point_y(DoodleHashCheekPair *cheek, guint side)
{
  if (side == DOODLEHASH_SIDE_LEFT)
    return cheek->left->connect_to_head_y;
  else
    return cheek->right->connect_to_head_y;
}

guint32 doodlehash_cheeks_get_width(DoodleHashCheekPair *cheek, guint side)
{
  return doodlehash_object_get_width (side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base);
}

guint32 doodlehash_cheeks_get_height(DoodleHashCheekPair *cheek, guint side)
{
  return doodlehash_object_get_height (side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base);
}

void doodlehash_cheeks_set_random_matching_colours(DoodleHashCheekPair *cheeks)
{
  doodlehash_object_set_random_matching_colours(cheeks->left->base, 
                                                cheeks->right->base,
                                                NULL);
}

void doodlehash_cheeks_set_random_colours(DoodleHashCheekPair *cheek, guint code)
{
  doodlehash_object_set_random_colours(cheek->left->base, code);
  doodlehash_object_set_random_colours(cheek->right->base, code);
}

void doodlehash_cheeks_set_colours(DoodleHashCheekPair *cheek, guint side, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base, code, s, f);
}

gchar *doodlehash_cheeks_get_fill_colour(DoodleHashCheekPair *cheek, guint side, guint code)
{
  return doodlehash_object_get_fill_colour(side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base, code);
}

gchar *doodlehash_cheeks_get_stroke_colour(DoodleHashCheekPair *cheek, guint side, guint code)
{
  return doodlehash_object_get_stroke_colour(side == DOODLEHASH_SIDE_LEFT ? cheek->left->base : cheek->right->base, code);
}

void doodlehash_cheeks_scale(DoodleHashCheekPair *ch, guint side, double scale)
{
  DoodleHashCheek *c = (side == DOODLEHASH_SIDE_LEFT) ? ch->left : ch->right;
  c->connect_to_head_x *= scale;
  c->connect_to_head_y *= scale;
  doodlehash_object_scale(c->base, scale);
}

void doodlehash_cheeks_free(DoodleHashCheekPair *c)
{
  if (c && c->left && c->left->base)
    doodlehash_object_free (c->left->base);
  if (c && c->right && c->right->base)
    doodlehash_object_free (c->right->base);
  if (c && c->right)
    g_free (c->right);
  if (c && c->left)
    g_free (c->left);
  g_free (c);
}
